"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getWorkflowToolType = void 0;
var _zod = require("@kbn/zod");
var _onechatCommon = require("@kbn/onechat-common");
var _onechatServer = require("@kbn/onechat-server");
var _execute_workflow = require("./execute_workflow");
var _generate_schema = require("./generate_schema");
var _schemas = require("./schemas");
var _validation = require("./validation");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getWorkflowToolType = ({
  workflowsManagement
}) => {
  // workflow plugin not present - we disable the workflow tool type
  if (!workflowsManagement) {
    return {
      toolType: _onechatCommon.ToolType.workflow,
      disabled: true
    };
  }
  return {
    toolType: _onechatCommon.ToolType.workflow,
    getDynamicProps: (config, {
      spaceId
    }) => {
      return {
        getHandler: () => {
          return async (params, {
            request
          }) => {
            const {
              management: workflowApi
            } = workflowsManagement;
            const workflowId = config.workflow_id;
            try {
              const workflowResults = await (0, _execute_workflow.executeWorkflow)({
                request,
                spaceId,
                workflowApi,
                workflowId,
                workflowParams: params
              });
              return {
                results: workflowResults
              };
            } catch (e) {
              return {
                results: [(0, _onechatServer.createErrorResult)({
                  message: `Error executing workflow: ${e}`,
                  metadata: {
                    workflowId
                  }
                })]
              };
            }
          };
        },
        getSchema: async () => {
          const workflow = await workflowsManagement.management.getWorkflow(config.workflow_id, spaceId);
          if (!workflow) {
            // if workflow is not accessible, error will be handled elsewhere
            return _zod.z.object({});
          }
          return (0, _generate_schema.generateSchema)({
            workflow
          });
        }
      };
    },
    createSchema: _schemas.configurationSchema,
    updateSchema: _schemas.configurationUpdateSchema,
    validateForCreate: async ({
      config,
      context: {
        spaceId
      }
    }) => {
      await (0, _validation.validateWorkflowId)({
        workflows: workflowsManagement,
        workflowId: config.workflow_id,
        spaceId
      });
      return config;
    },
    validateForUpdate: async ({
      update,
      current,
      context: {
        spaceId
      }
    }) => {
      const mergedConfig = {
        ...current,
        ...update
      };
      await (0, _validation.validateWorkflowId)({
        workflows: workflowsManagement,
        workflowId: mergedConfig.workflow_id,
        spaceId
      });
      return mergedConfig;
    }
  };
};
exports.getWorkflowToolType = getWorkflowToolType;