"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.executeEsqlTool = void 0;
var _zod = require("@kbn/zod");
var _onechatCommon = require("@kbn/onechat-common");
var _esql = require("@kbn/onechat-genai-utils/tools/utils/esql");
var _tool_result = require("@kbn/onechat-common/tools/tool_result");
var _tools = require("@kbn/onechat-server/tools");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const executeEsqlToolSchema = _zod.z.object({
  query: _zod.z.string().describe('The ES|QL query to execute')
});
const executeEsqlTool = () => {
  return {
    id: _onechatCommon.platformCoreTools.executeEsql,
    type: _onechatCommon.ToolType.builtin,
    description: `Execute an ES|QL query and return the results in a tabular format.

**IMPORTANT**: This tool only **runs** queries; it does not write them.
Think of this as the final step after a query has been prepared.

You **must** get the query from one of two sources before calling this tool:
1.  The output of the \`${_onechatCommon.platformCoreTools.generateEsql}\` tool (if the tool is available).
2.  A verbatim query provided directly by the user.

Under no circumstances should you invent, guess, or modify a query yourself for this tool.
If you need a query, use the \`${_onechatCommon.platformCoreTools.generateEsql}\` tool first.`,
    schema: executeEsqlToolSchema,
    handler: async ({
      query: esqlQuery
    }, {
      esClient
    }) => {
      const result = await (0, _esql.executeEsql)({
        query: esqlQuery,
        esClient: esClient.asCurrentUser
      });
      return {
        results: [{
          type: _tool_result.ToolResultType.query,
          data: {
            esql: esqlQuery
          }
        }, {
          tool_result_id: (0, _tools.getToolResultId)(),
          type: _tool_result.ToolResultType.tabularData,
          data: {
            source: 'esql',
            query: esqlQuery,
            columns: result.columns,
            values: result.values
          }
        }]
      };
    },
    tags: []
  };
};
exports.executeEsqlTool = executeEsqlTool;