"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getFoldingRanges = void 0;
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const getOpeningLineRegex = openingMarker => {
  // Opening parentheses can only be preceded by a colon or nothing
  // This ensures that it's not between quotes
  const regExStr = `^(.*:\\s*)?\\${openingMarker}$`;
  return new RegExp(regExStr);
};
const getClosingLineRegex = closingMarker => {
  // Closing marker can only be followed by a comma or nothing
  // This ensures that it's not between quotes
  const regExStr = `^\\${closingMarker}\\s*(,)?$`;
  return new RegExp(regExStr);
};

// A regex that matches a line containing a complete triple-quote string
const inlineTripleQuoteString = /^.*?"""(.*?)""".*$/m;
const getFoldingRanges = (lines, openingMarker, closingMarker) => {
  const ranges = [];
  const stack = [];
  const openingLineRegex = getOpeningLineRegex(openingMarker);
  const closingLineRegex = getClosingLineRegex(closingMarker);
  let insideTripleQuotes = false;
  for (let i = 0; i < lines.length; i++) {
    const lineContent = lines[i].trim();
    if (lineContent.includes('"""') && !inlineTripleQuoteString.test(lineContent)) {
      insideTripleQuotes = !insideTripleQuotes;
    }
    if (insideTripleQuotes) {
      // If we are inside a multi-line triple-quote string, ignore opening/closing markers
      continue;
    }
    if (openingLineRegex.test(lineContent)) {
      stack.push(i + 1); // Line numbers start from 1 so we need to add 1 to the current index
    } else if (closingLineRegex.test(lineContent)) {
      const start = stack.pop();
      if (start) {
        ranges.push({
          start,
          end: i
        });
      }
    }
  }
  return ranges;
};
exports.getFoldingRanges = getFoldingRanges;