"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ignoreErrorsMap = void 0;
exports.validateQuery = validateQuery;
var _esqlAst = require("@kbn/esql-ast");
var _utils = require("@kbn/esql-ast/src/definitions/utils");
var _resources_helpers = require("../shared/resources_helpers");
var _resources = require("./resources");
var _helpers = require("./helpers");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * ES|QL validation public API
 * It takes a query string and returns a list of messages (errors and warnings) after validate
 * The astProvider is optional, but if not provided the default one from '@kbn/esql-validation-autocomplete' will be used.
 * This is useful for async loading the ES|QL parser and reduce the bundle size, or to swap grammar version.
 * As for the callbacks, while optional, the validation function will selectively ignore some errors types based on each callback missing.
 */
async function validateQuery(queryString, options = {}, callbacks) {
  const result = await validateAst(queryString, callbacks);
  // early return if we do not want to ignore errors
  if (!options.ignoreOnMissingCallbacks) {
    return result;
  }
  const finalCallbacks = callbacks || {};
  const errorTypoesToIgnore = Object.entries(ignoreErrorsMap).reduce((acc, [key, errorCodes]) => {
    if (!(key in finalCallbacks) || key in finalCallbacks && finalCallbacks[key] == null) {
      for (const e of errorCodes) {
        acc[e] = true;
      }
    }
    return acc;
  }, {});
  const filteredErrors = result.errors.filter(error => {
    if ('severity' in error) {
      return true;
    }
    return !errorTypoesToIgnore[error.code];
  }).map(error => 'severity' in error ? {
    text: error.message,
    code: error.code,
    type: 'error',
    location: {
      min: error.startColumn,
      max: error.endColumn
    }
  } : error);
  return {
    errors: filteredErrors,
    warnings: result.warnings
  };
}

/**
 * @internal
 */
const ignoreErrorsMap = exports.ignoreErrorsMap = {
  getColumnsFor: ['unknownColumn', 'unsupportedFieldType'],
  getSources: ['unknownIndex'],
  getPolicies: ['unknownPolicy'],
  getPreferences: [],
  getFieldsMetadata: [],
  getVariables: [],
  canSuggestVariables: [],
  getJoinIndices: ['invalidJoinIndex'],
  getTimeseriesIndices: ['unknownIndex'],
  getEditorExtensions: [],
  getInferenceEndpoints: [],
  getLicense: [],
  getActiveProduct: [],
  canCreateLookupIndex: []
};

/**
 * This function will perform an high level validation of the
 * query AST. An initial syntax validation is already performed by the parser
 * while here it can detect things like function names, types correctness and potential warnings
 * @param ast A valid AST data structure
 */
async function validateAst(queryString, callbacks) {
  var _callbacks$getJoinInd, _callbacks$getLicense;
  const messages = [];
  const parsingResult = _esqlAst.EsqlQuery.fromSrc(queryString);
  const rootCommands = parsingResult.ast.commands;
  const [sources, availablePolicies, joinIndices] = await Promise.all([
  // retrieve the list of available sources
  (0, _resources.retrieveSources)(rootCommands, callbacks),
  // retrieve available policies (if an enrich command has been defined)
  (0, _resources.retrievePolicies)(rootCommands, callbacks), // retrieve indices for join command
  callbacks === null || callbacks === void 0 ? void 0 : (_callbacks$getJoinInd = callbacks.getJoinIndices) === null || _callbacks$getJoinInd === void 0 ? void 0 : _callbacks$getJoinInd.call(callbacks)]);
  const sourceQuery = queryString.split('|')[0];
  const sourceFields = await new _resources_helpers.QueryColumns(_esqlAst.EsqlQuery.fromSrc(sourceQuery).ast, sourceQuery, callbacks).asMap();

  // TODO move into the loop?
  messages.push(...validateUnsupportedTypeFields(sourceFields, rootCommands));
  const license = await (callbacks === null || callbacks === void 0 ? void 0 : (_callbacks$getLicense = callbacks.getLicense) === null || _callbacks$getLicense === void 0 ? void 0 : _callbacks$getLicense.call(callbacks));
  const hasMinimumLicenseRequired = license === null || license === void 0 ? void 0 : license.hasAtLeast;

  /**
   * Even though we are validating single commands, we work with subqueries.
   *
   * The reason is that building the list of columns available in each command requires
   * the full command subsequence that precedes that command.
   */
  const subqueries = (0, _helpers.getSubqueriesToValidate)(rootCommands);
  for (const subquery of subqueries) {
    const subqueryUpToThisCommand = {
      ...subquery,
      commands: subquery.commands.slice(0, -1)
    };
    const columns = await new _resources_helpers.QueryColumns(subqueryUpToThisCommand, queryString, callbacks).asMap();
    const references = {
      sources,
      columns,
      policies: availablePolicies,
      query: queryString,
      joinIndices: (joinIndices === null || joinIndices === void 0 ? void 0 : joinIndices.indices) || []
    };
    const commandMessages = validateCommand(subquery.commands[subquery.commands.length - 1], references, rootCommands, {
      ...callbacks,
      hasMinimumLicenseRequired
    });
    messages.push(...commandMessages);
  }
  const parserErrors = parsingResult.errors;

  /**
   * Some changes to the grammar deleted the literal names for some tokens.
   * This is a workaround to restore the literals that were lost.
   *
   * See https://github.com/elastic/elasticsearch/pull/124177 for context.
   */
  for (const error of parserErrors) {
    error.message = error.message.replace(/\bLP\b/, "'('");
    error.message = error.message.replace(/\bOPENING_BRACKET\b/, "'['");
  }
  return {
    errors: [...parserErrors, ...messages.filter(({
      type
    }) => type === 'error')],
    warnings: messages.filter(({
      type
    }) => type === 'warning')
  };
}
function validateCommand(command, references, ast, callbacks) {
  const messages = [];
  if (command.incomplete) {
    return messages;
  }
  // do not check the command exists, the grammar is already picking that up
  const commandDefinition = _esqlAst.esqlCommandRegistry.getCommandByName(command.name);
  if (!commandDefinition) {
    return messages;
  }

  // Check license requirements for the command
  if (callbacks !== null && callbacks !== void 0 && callbacks.hasMinimumLicenseRequired) {
    const license = commandDefinition.metadata.license;
    if (license && !callbacks.hasMinimumLicenseRequired(license.toLowerCase())) {
      messages.push((0, _utils.getMessageFromId)({
        messageId: 'licenseRequired',
        values: {
          name: command.name.toUpperCase(),
          requiredLicense: license.toUpperCase()
        },
        locations: command.location
      }));
    }
  }
  const context = {
    columns: references.columns,
    policies: references.policies,
    sources: [...references.sources].map(source => ({
      name: source
    })),
    joinSources: references.joinIndices
  };
  if (commandDefinition.methods.validate) {
    messages.push(...commandDefinition.methods.validate(command, ast, context, callbacks));
  }

  // no need to check for mandatory options passed
  // as they are already validated at syntax level
  return messages;
}
function validateUnsupportedTypeFields(fields, ast) {
  const usedColumnsInQuery = [];
  (0, _esqlAst.walk)(ast, {
    visitColumn: node => usedColumnsInQuery.push(node.name)
  });
  const messages = [];
  for (const column of usedColumnsInQuery) {
    if (fields.has(column) && fields.get(column).type === 'unsupported') {
      messages.push((0, _utils.getMessageFromId)({
        messageId: 'unsupportedFieldType',
        values: {
          field: column
        },
        locations: {
          min: 1,
          max: 1
        }
      }));
    }
  }
  return messages;
}