"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.convertTimeseriesCommandToFrom = convertTimeseriesCommandToFrom;
exports.findClosestColumn = findClosestColumn;
exports.getESQLQueryVariables = exports.getCategorizeField = exports.getCategorizeColumns = exports.getArgsFromRenameFunction = exports.fixESQLQueryWithVariables = void 0;
exports.getIndexPatternFromESQLQuery = getIndexPatternFromESQLQuery;
exports.getKqlSearchQueries = void 0;
exports.getLimitFromESQLQuery = getLimitFromESQLQuery;
exports.getQueryUpToCursor = exports.getQueryColumnsFromESQLQuery = void 0;
exports.getRemoteClustersFromESQLQuery = getRemoteClustersFromESQLQuery;
exports.getValuesFromQueryField = exports.getTimeFieldFromESQLQuery = void 0;
exports.hasTransformationalCommand = hasTransformationalCommand;
exports.prettifyQuery = exports.mapVariableToColumn = exports.isQueryWrappedByPipes = void 0;
exports.removeDropCommandsFromESQLQuery = removeDropCommandsFromESQLQuery;
exports.retrieveMetadataColumns = void 0;
var _esqlAst = require("@kbn/esql-ast");
var _esqlTypes = require("@kbn/esql-types");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const DEFAULT_ESQL_LIMIT = 1000;

// retrieves the index pattern from the aggregate query for ES|QL using ast parsing
function getIndexPatternFromESQLQuery(esql) {
  var _sourceCommand$args;
  const {
    ast
  } = (0, _esqlAst.parse)(esql);
  const sourceCommand = ast.find(({
    name
  }) => ['from', 'ts'].includes(name));
  const args = (_sourceCommand$args = sourceCommand === null || sourceCommand === void 0 ? void 0 : sourceCommand.args) !== null && _sourceCommand$args !== void 0 ? _sourceCommand$args : [];
  const indices = args.filter(arg => arg.sourceType === 'index');
  return indices === null || indices === void 0 ? void 0 : indices.map(index => index.name).join(',');
}
function getRemoteClustersFromESQLQuery(esql) {
  var _sourceCommand$args2;
  if (!esql) return undefined;
  const {
    root
  } = _esqlAst.Parser.parse(esql);
  const sourceCommand = root.commands.find(({
    name
  }) => ['from', 'ts'].includes(name));
  const args = (_sourceCommand$args2 = sourceCommand === null || sourceCommand === void 0 ? void 0 : sourceCommand.args) !== null && _sourceCommand$args2 !== void 0 ? _sourceCommand$args2 : [];
  const clustersSet = new Set();

  // Handle sources with explicit prefix (e.g., FROM cluster1:index1)
  args.filter(arg => arg.prefix).forEach(arg => {
    var _arg$prefix;
    if ((_arg$prefix = arg.prefix) !== null && _arg$prefix !== void 0 && _arg$prefix.value) {
      clustersSet.add(arg.prefix.value);
    }
  });

  // Handle sources without prefix that might contain cluster:index patterns
  // This includes quoted sources like "cluster1:index1,cluster2:index2"
  args.filter(arg => !arg.prefix).forEach(arg => {
    // Split by comma to handle cases like "cluster1:index1,cluster2:index2"
    const indices = arg.name.split(',');
    indices.forEach(index => {
      const trimmedIndex = index.trim();
      const colonIndex = trimmedIndex.indexOf(':');
      // Only add if there's a valid cluster:index pattern
      if (colonIndex > 0 && colonIndex < trimmedIndex.length - 1) {
        const clusterName = trimmedIndex.substring(0, colonIndex);
        clustersSet.add(clusterName);
      }
    });
  });
  return clustersSet.size > 0 ? [...clustersSet] : undefined;
}

// For ES|QL we consider stats and keep transformational command
// The metrics command too but only if it aggregates
function hasTransformationalCommand(esql) {
  const transformationalCommands = ['stats', 'keep'];
  const {
    ast
  } = (0, _esqlAst.parse)(esql);
  const hasAtLeastOneTransformationalCommand = transformationalCommands.some(command => ast.find(({
    name
  }) => name === command));
  return hasAtLeastOneTransformationalCommand;
}
function getLimitFromESQLQuery(esql) {
  const {
    ast
  } = (0, _esqlAst.parse)(esql);
  const limitCommands = ast.filter(({
    name
  }) => name === 'limit');
  if (!limitCommands || !limitCommands.length) {
    return DEFAULT_ESQL_LIMIT;
  }
  const limits = [];
  (0, _esqlAst.walk)(ast, {
    visitLiteral: node => {
      if (!isNaN(Number(node.value))) {
        limits.push(Number(node.value));
      }
    }
  });
  if (!limits.length) {
    return DEFAULT_ESQL_LIMIT;
  }

  // ES returns always the smallest limit
  return Math.min(...limits);
}
function removeDropCommandsFromESQLQuery(esql) {
  const pipes = (esql || '').split('|');
  return pipes.filter(statement => !/DROP\s/i.test(statement)).join('|');
}

/**
 * Converts timeseries (TS) commands to FROM commands in an ES|QL query
 * @param esql - The ES|QL query string
 * @returns The modified query with TS commands converted to FROM commands
 */
function convertTimeseriesCommandToFrom(esql) {
  const {
    root
  } = _esqlAst.Parser.parse(esql || '');
  const timeseriesCommand = _esqlAst.Walker.commands(root).find(({
    name
  }) => name === 'ts');
  if (!timeseriesCommand) return esql || '';
  const fromCommand = {
    ...timeseriesCommand,
    name: 'from'
  };

  // Replace the ts command with the from command in the commands array
  const newCommands = root.commands.map(command => command === timeseriesCommand ? fromCommand : command);
  const newRoot = {
    ...root,
    commands: newCommands
  };
  return _esqlAst.BasicPrettyPrinter.print(newRoot);
}

/**
 * When the ?_tstart and ?_tend params are used, we want to retrieve the timefield from the query.
 * @param esql:string
 * @returns string
 */
const getTimeFieldFromESQLQuery = esql => {
  const {
    ast
  } = (0, _esqlAst.parse)(esql);
  const functions = [];
  (0, _esqlAst.walk)(ast, {
    visitFunction: node => functions.push(node)
  });
  const params = _esqlAst.Walker.params(ast);
  const timeNamedParam = params.find(param => param.value === '_tstart' || param.value === '_tend');
  if (!timeNamedParam || !functions.length) {
    return undefined;
  }
  const allFunctionsWithNamedParams = functions.filter(({
    location
  }) => location.min <= timeNamedParam.location.min && location.max >= timeNamedParam.location.max);
  if (!allFunctionsWithNamedParams.length) {
    return undefined;
  }
  const lowLevelFunction = allFunctionsWithNamedParams[allFunctionsWithNamedParams.length - 1];
  let columnName;
  lowLevelFunction.args.some(arg => {
    const argument = arg;
    if (argument.type === 'column') {
      columnName = argument.name;
      return true;
    }
    if (argument.type === 'inlineCast' && argument.value.type === 'column') {
      columnName = argument.value.name;
      return true;
    }
    return false;
  });
  return columnName;
};
exports.getTimeFieldFromESQLQuery = getTimeFieldFromESQLQuery;
const getKqlSearchQueries = esql => {
  const {
    ast
  } = (0, _esqlAst.parse)(esql);
  const functions = [];
  (0, _esqlAst.walk)(ast, {
    visitFunction: node => functions.push(node)
  });
  const searchFunctions = functions.filter(({
    name
  }) => name === 'kql');
  return searchFunctions.map(func => {
    if (func.args.length > 0 && (0, _esqlAst.isStringLiteral)(func.args[0])) {
      return func.args[0].valueUnquoted.trim();
    }
    return '';
  }).filter(query => query !== '');
};
exports.getKqlSearchQueries = getKqlSearchQueries;
const isQueryWrappedByPipes = query => {
  const {
    ast
  } = (0, _esqlAst.parse)(query);
  const numberOfCommands = ast.length;
  const pipesWithNewLine = query.split('\n  |');
  return numberOfCommands === (pipesWithNewLine === null || pipesWithNewLine === void 0 ? void 0 : pipesWithNewLine.length);
};
exports.isQueryWrappedByPipes = isQueryWrappedByPipes;
const prettifyQuery = src => {
  const {
    root
  } = _esqlAst.Parser.parse(src, {
    withFormatting: true
  });
  return _esqlAst.WrappingPrettyPrinter.print(root, {
    multiline: true
  });
};
exports.prettifyQuery = prettifyQuery;
const retrieveMetadataColumns = esql => {
  var _metadataOptions$args;
  const {
    ast
  } = (0, _esqlAst.parse)(esql);
  const options = [];
  (0, _esqlAst.walk)(ast, {
    visitCommandOption: node => options.push(node)
  });
  const metadataOptions = options.find(({
    name
  }) => name === 'metadata');
  return (_metadataOptions$args = metadataOptions === null || metadataOptions === void 0 ? void 0 : metadataOptions.args.map(column => column.name)) !== null && _metadataOptions$args !== void 0 ? _metadataOptions$args : [];
};
exports.retrieveMetadataColumns = retrieveMetadataColumns;
const getQueryColumnsFromESQLQuery = esql => {
  const {
    root
  } = (0, _esqlAst.parse)(esql);
  const columns = [];
  (0, _esqlAst.walk)(root, {
    visitColumn: node => columns.push(node)
  });
  return columns.map(column => column.name);
};
exports.getQueryColumnsFromESQLQuery = getQueryColumnsFromESQLQuery;
const getESQLQueryVariables = esql => {
  const {
    root
  } = (0, _esqlAst.parse)(esql);
  const usedVariablesInQuery = _esqlAst.Walker.params(root);
  return usedVariablesInQuery.map(v => v.text.replace(/^\?+/, ''));
};

/**
 * This function is used to map the variables to the columns in the datatable
 * @param esql:string
 * @param variables:ESQLControlVariable[]
 * @param columns:DatatableColumn[]
 * @returns DatatableColumn[]
 */
exports.getESQLQueryVariables = getESQLQueryVariables;
const mapVariableToColumn = (esql, variables, columns) => {
  if (!variables.length) {
    return columns;
  }
  const usedVariablesInQuery = getESQLQueryVariables(esql);
  const uniqueVariablesInQyery = new Set(usedVariablesInQuery);
  columns.map(column => {
    if (variables.some(variable => variable.value === column.id)) {
      var _variable$key;
      const potentialColumnVariables = variables.filter(variable => variable.value === column.id);
      const variable = potentialColumnVariables.find(v => uniqueVariablesInQyery.has(v.key));
      column.variable = (_variable$key = variable === null || variable === void 0 ? void 0 : variable.key) !== null && _variable$key !== void 0 ? _variable$key : '';
    }
  });
  return columns;
};
exports.mapVariableToColumn = mapVariableToColumn;
const getQueryUpToCursor = (queryString, cursorPosition) => {
  var _cursorPosition$lineN, _cursorPosition$colum;
  const lines = queryString.split('\n');
  const lineNumber = (_cursorPosition$lineN = cursorPosition === null || cursorPosition === void 0 ? void 0 : cursorPosition.lineNumber) !== null && _cursorPosition$lineN !== void 0 ? _cursorPosition$lineN : lines.length;
  const column = (_cursorPosition$colum = cursorPosition === null || cursorPosition === void 0 ? void 0 : cursorPosition.column) !== null && _cursorPosition$colum !== void 0 ? _cursorPosition$colum : lines[lineNumber - 1].length;

  // Handle the case where the cursor is within the first line
  if (lineNumber === 1) {
    return lines[0].slice(0, column);
  }

  // Get all lines up to the specified line number (exclusive of the current line)
  const previousLines = lines.slice(0, lineNumber - 1).join('\n');
  const currentLine = lines[lineNumber - 1].slice(0, column);

  // Join the previous lines and the partial current line
  return previousLines + '\n' + currentLine;
};
exports.getQueryUpToCursor = getQueryUpToCursor;
const hasQuestionMarkAtEndOrSecondLastPosition = queryString => {
  if (typeof queryString !== 'string' || queryString.length === 0) {
    return false;
  }
  const lastChar = queryString.slice(-1);
  const secondLastChar = queryString.slice(-2, -1);
  return lastChar === '?' || secondLastChar === '?';
};

/**
 * Finds the column closest to the given cursor position within an array of columns.
 *
 * @param columns An array of ES|QL columns.
 * @param cursorPosition The current cursor position.
 * @returns The column object closest to the cursor, or null if the columns array is empty.
 */
function findClosestColumn(columns, cursorPosition) {
  if (columns.length === 0) {
    return undefined;
  }
  if (!cursorPosition) {
    return columns[0]; // If no cursor position is provided, return the first column
  }
  const cursorCol = cursorPosition.column;
  let closestColumn;
  let minDistance = Infinity;
  for (const column of columns) {
    const columnMin = column.location.min;
    const columnMax = column.location.max;

    // If the cursor is within the column's range, it's the closest
    if (cursorCol >= columnMin && cursorCol <= columnMax) {
      return column;
    }

    // Calculate distance from the cursor to the nearest edge of the column
    let distance;
    if (cursorCol < columnMin) {
      distance = columnMin - cursorCol; // Cursor is to the left of the column
    } else {
      distance = cursorCol - columnMax; // Cursor is to the right of the column
    }
    if (distance < minDistance) {
      minDistance = distance;
      closestColumn = column;
    }
  }
  return closestColumn;
}
const getValuesFromQueryField = (queryString, cursorPosition) => {
  const queryInCursorPosition = getQueryUpToCursor(queryString, cursorPosition);
  if (hasQuestionMarkAtEndOrSecondLastPosition(queryInCursorPosition)) {
    return undefined;
  }
  const validQuery = `${queryInCursorPosition} ""`;
  const {
    root
  } = (0, _esqlAst.parse)(validQuery);
  const lastCommand = root.commands[root.commands.length - 1];
  const columns = [];
  (0, _esqlAst.walk)(lastCommand, {
    visitColumn: node => columns.push(node)
  });
  const column = findClosestColumn(columns, cursorPosition);
  if (column && column.name && column.name !== '*') {
    return `${column.name}`;
  }
};

// this is for backward compatibility, if the query is of fields or functions type
// and the query is not set with ?? in the query, we should set it
// https://github.com/elastic/elasticsearch/pull/122459
exports.getValuesFromQueryField = getValuesFromQueryField;
const fixESQLQueryWithVariables = (queryString, esqlVariables) => {
  const currentVariables = getESQLQueryVariables(queryString);
  if (!currentVariables.length) {
    return queryString;
  }

  // filter out the variables that are not used in the query
  // and that they are not of type FIELDS or FUNCTIONS
  const identifierTypeVariables = esqlVariables === null || esqlVariables === void 0 ? void 0 : esqlVariables.filter(variable => currentVariables.includes(variable.key) && (variable.type === _esqlTypes.ESQLVariableType.FIELDS || variable.type === _esqlTypes.ESQLVariableType.FUNCTIONS));

  // check if they are set with ?? or ? in the query
  // replace only if there is only one ? in front of the variable
  if (identifierTypeVariables !== null && identifierTypeVariables !== void 0 && identifierTypeVariables.length) {
    identifierTypeVariables.forEach(variable => {
      const regex = new RegExp(`(?<!\\?)\\?${variable.key}`);
      queryString = queryString.replace(regex, `??${variable.key}`);
    });
    return queryString;
  }
  return queryString;
};
exports.fixESQLQueryWithVariables = fixESQLQueryWithVariables;
const getCategorizeColumns = esql => {
  const {
    root
  } = (0, _esqlAst.parse)(esql);
  const statsCommand = root.commands.find(({
    name
  }) => name === 'stats');
  if (!statsCommand) {
    return [];
  }
  const options = [];
  const columns = [];
  (0, _esqlAst.walk)(statsCommand, {
    visitCommandOption: node => options.push(node)
  });
  const statsByOptions = options.find(({
    name
  }) => name === 'by');

  // categorize is part of the stats by command
  if (!statsByOptions) {
    return [];
  }
  const categorizeOptions = statsByOptions.args.filter(arg => {
    return arg.text.toLowerCase().indexOf('categorize') !== -1;
  });
  if (categorizeOptions.length) {
    categorizeOptions.forEach(arg => {
      // ... STATS ... BY CATEGORIZE(field)
      if ((0, _esqlAst.isFunctionExpression)(arg) && arg.name === 'categorize') {
        columns.push(arg.text);
      } else {
        // ... STATS ... BY pattern = CATEGORIZE(field)
        const columnArgs = arg.args.filter(a => (0, _esqlAst.isColumn)(a));
        columnArgs.forEach(c => columns.push(c.name));
      }
    });
  }

  // If there is a rename command, we need to check if the column is renamed
  const renameCommand = root.commands.find(({
    name
  }) => name === 'rename');
  if (!renameCommand) {
    return columns;
  }
  const renameFunctions = [];
  (0, _esqlAst.walk)(renameCommand, {
    visitFunction: node => renameFunctions.push(node)
  });
  renameFunctions.forEach(renameFunction => {
    const {
      original,
      renamed
    } = getArgsFromRenameFunction(renameFunction);
    const oldColumn = original.name;
    const newColumn = renamed.name;
    if (columns.includes(oldColumn)) {
      columns[columns.indexOf(oldColumn)] = newColumn;
    }
  });
  return columns;
};

/**
 * Extracts the original and renamed columns from a rename function.
 * RENAME original AS renamed Vs RENAME renamed = original
 * @param renameFunction
 */
exports.getCategorizeColumns = getCategorizeColumns;
const getArgsFromRenameFunction = renameFunction => {
  if (renameFunction.name === 'as') {
    return {
      original: renameFunction.args[0],
      renamed: renameFunction.args[1]
    };
  }
  return {
    original: renameFunction.args[1],
    renamed: renameFunction.args[0]
  };
};

/**
 * Extracts the fields used in the CATEGORIZE function from an ESQL query.
 * @param esql: string - The ESQL query string
 */
exports.getArgsFromRenameFunction = getArgsFromRenameFunction;
const getCategorizeField = esql => {
  const {
    root
  } = (0, _esqlAst.parse)(esql);
  const columns = [];
  const functions = _esqlAst.Walker.matchAll(root.commands, {
    type: 'function',
    name: 'categorize'
  });
  if (functions.length) {
    functions.forEach(func => {
      for (const arg of func.args) if ((0, _esqlAst.isColumn)(arg)) columns.push(arg.name);
    });
    return columns;
  }
  return columns;
};
exports.getCategorizeField = getCategorizeField;