"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getExtendedChangePoint = void 0;
var _d3Array = require("d3-array");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Calculates and returns an extended change point range based on the specified change point timestamp.
 *
 * @param buckets - An object where keys are bucket timestamps as strings
 *                  and values are numeric values associated with each bucket.
 * @param changePointTs - The timestamp of the change point as a number. This timestamp must
 *                        be one of the keys in the `buckets` object.
 * @returns An object containing two properties: `startTs` and `endTs`.
 */
const getExtendedChangePoint = (buckets, changePointTs) => {
  var _mean;
  const bucketKeys = Object.keys(buckets);
  const bucketValues = Object.values(buckets);
  const meanValue = Math.round((_mean = (0, _d3Array.mean)(bucketValues)) !== null && _mean !== void 0 ? _mean : 0);
  const cpIndex = bucketKeys.findIndex(d => +d === changePointTs);
  const cpValue = buckets[changePointTs];
  let lIndex = cpIndex - 1;
  let uIndex = cpIndex + 1;
  while (lIndex >= 0 && Math.abs(bucketValues[lIndex] - meanValue) > Math.abs(bucketValues[lIndex] - cpValue)) {
    lIndex--;
  }
  while (uIndex < bucketValues.length && Math.abs(bucketValues[uIndex] - meanValue) > Math.abs(bucketValues[uIndex] - cpValue)) {
    uIndex++;
  }
  return {
    startTs: +bucketKeys[lIndex],
    endTs: +bucketKeys[uIndex]
  };
};
exports.getExtendedChangePoint = getExtendedChangePoint;