"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerCustomScriptsRoute = exports.getCustomScriptsRouteHandler = void 0;
var _custom_http_request_error = require("../../../utils/custom_http_request_error");
var _is_response_action_supported = require("../../../../common/endpoint/service/response_actions/is_response_action_supported");
var _services = require("../../services");
var _error_handler = require("../error_handler");
var _constants = require("../../../../common/endpoint/constants");
var _get_custom_scripts_route = require("../../../../common/api/endpoint/custom_scripts/get_custom_scripts_route");
var _with_endpoint_authz = require("../with_endpoint_authz");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Registers the custom scripts route
 * @param router - Security solution plugin router
 * @param endpointContext - Endpoint app context
 */
const registerCustomScriptsRoute = (router, endpointContext) => {
  router.versioned.get({
    access: 'internal',
    path: _constants.CUSTOM_SCRIPTS_ROUTE,
    security: {
      authz: {
        requiredPrivileges: ['securitySolution']
      }
    },
    options: {
      authRequired: true
    }
  }).addVersion({
    version: '1',
    validate: {
      request: _get_custom_scripts_route.CustomScriptsRequestSchema
    }
  }, (0, _with_endpoint_authz.withEndpointAuthz)({
    all: ['canWriteExecuteOperations']
  }, endpointContext.logFactory.get('customScriptsRoute'), getCustomScriptsRouteHandler(endpointContext)));
};

/**
 * Creates a handler for the custom scripts route
 * @param endpointContext - Endpoint app context
 * @returns Request handler for custom scripts
 */
exports.registerCustomScriptsRoute = registerCustomScriptsRoute;
const getCustomScriptsRouteHandler = endpointContext => {
  const logger = endpointContext.logFactory.get('customScriptsRouteHandler');
  return async (context, request, response) => {
    const {
      agentType = 'endpoint',
      ...otherQueryOptions
    } = request.query;
    logger.debug(`Retrieving custom scripts for: agentType ${agentType}`);
    if (!(0, _is_response_action_supported.isActionSupportedByAgentType)(agentType, 'runscript', 'manual') || agentType === 'sentinel_one' && !endpointContext.experimentalFeatures.responseActionsSentinelOneRunScriptEnabled) {
      return (0, _error_handler.errorHandler)(logger, response, new _custom_http_request_error.CustomHttpRequestError(`Agent type [${agentType}] does not support 'runscript' response action`, 400));
    }
    try {
      const coreContext = await context.core;
      const user = coreContext.security.authc.getCurrentUser();
      const esClient = coreContext.elasticsearch.client.asInternalUser;
      const connectorActions = (await context.actions).getActionsClient();
      const spaceId = (await context.securitySolution).getSpaceId();
      const responseActionsClient = (0, _services.getResponseActionsClient)(agentType, {
        esClient,
        spaceId,
        endpointService: endpointContext.service,
        username: (user === null || user === void 0 ? void 0 : user.username) || 'unknown',
        connectorActions: new _services.NormalizedExternalConnectorClient(connectorActions, logger)
      });
      const data = await responseActionsClient.getCustomScripts(otherQueryOptions);
      return response.ok({
        body: data
      });
    } catch (e) {
      return (0, _error_handler.errorHandler)(logger, response, e);
    }
  };
};
exports.getCustomScriptsRouteHandler = getCustomScriptsRouteHandler;