"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.TopNType = exports.StackTracesDisplayOption = void 0;
exports.decodeStackTraceResponse = decodeStackTraceResponse;
exports.getFieldNameForTopNType = getFieldNameForTopNType;
exports.makeFrameID = void 0;
var _elasticsearch = require("./elasticsearch");
var _profiling = require("./profiling");
var _utils = require("./utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/** Profiling status response */

/** Profiling stacktrace */

/** Decoded stack trace response */

/**
 * Generate Frame ID
 * @param frameID string
 * @param n number
 * @returns string
 */
const makeFrameID = (frameID, n) => {
  return n === 0 ? frameID : frameID + ';' + n.toString();
};

// createInlineTrace builds a new StackTrace with inline frames.
exports.makeFrameID = makeFrameID;
const createInlineTrace = (trace, frames, showErrorFrames) => {
  // The arrays need to be extended with the inline frame information.
  const frameIDs = [];
  const fileIDs = [];
  const addressOrLines = [];
  const typeIDs = [];

  // Error frames only appear as first frame in a stacktrace.
  const start = !showErrorFrames && trace.frame_ids.length > 0 && (0, _profiling.isErrorFrame)(trace.type_ids[0]) ? 1 : 0;
  for (let i = start; i < trace.frame_ids.length; i++) {
    const frameID = trace.frame_ids[i];
    frameIDs.push(frameID);
    fileIDs.push(trace.file_ids[i]);
    addressOrLines.push(trace.address_or_lines[i]);
    typeIDs.push(trace.type_ids[i]);
    for (let j = 1;; j++) {
      const inlineID = makeFrameID(frameID, j);
      const frame = frames.get(inlineID);
      if (!frame) {
        break;
      }
      frameIDs.push(inlineID);
      fileIDs.push(trace.file_ids[i]);
      addressOrLines.push(trace.address_or_lines[i]);
      typeIDs.push(trace.type_ids[i]);
    }
  }
  return {
    FrameIDs: frameIDs,
    FileIDs: fileIDs,
    AddressOrLines: addressOrLines,
    Types: typeIDs,
    selfAnnualCO2Kgs: (0, _utils.convertTonsToKgs)(trace.annual_co2_tons),
    selfAnnualCostUSD: trace.annual_costs_usd,
    Count: trace.count
  };
};

/**
 * Decodes stack trace response
 * @param response StackTraceResponse
 * @param showErrorFrames
 * @returns DecodedStackTraceResponse
 */
function decodeStackTraceResponse(response, showErrorFrames) {
  const stackTraceEvents = new Map();
  for (const [key, value] of Object.entries((_response$stack_trace = response.stack_trace_events) !== null && _response$stack_trace !== void 0 ? _response$stack_trace : {})) {
    var _response$stack_trace;
    stackTraceEvents.set(key, value);
  }
  const stackFrames = new Map();
  for (const [frameID, frame] of Object.entries((_response$stack_frame = response.stack_frames) !== null && _response$stack_frame !== void 0 ? _response$stack_frame : {})) {
    var _response$stack_frame;
    // Each field in a stackframe is represented by an array. This is
    // necessary to support inline frames.
    //
    // We store the inlined frames with a modified (and unique) ID.
    // We can do so since we don't display the frame IDs.
    for (let i = 0; i < frame.function_name.length; i++) {
      stackFrames.set(makeFrameID(frameID, i), {
        FileName: frame.file_name[i],
        FunctionName: frame.function_name[i],
        FunctionOffset: frame.function_offset[i],
        LineNumber: frame.line_number[i],
        Inline: i > 0
      });
    }
  }
  const stackTraces = new Map();
  for (const [traceID, trace] of Object.entries((_response$stack_trace2 = response.stack_traces) !== null && _response$stack_trace2 !== void 0 ? _response$stack_trace2 : {})) {
    var _response$stack_trace2;
    stackTraces.set(traceID, createInlineTrace(trace, stackFrames, showErrorFrames));
  }
  const executables = new Map();
  for (const [key, value] of Object.entries((_response$executables = response.executables) !== null && _response$executables !== void 0 ? _response$executables : {})) {
    var _response$executables;
    executables.set(key, {
      FileName: value
    });
  }
  return {
    events: stackTraceEvents,
    stackTraces,
    stackFrames,
    executables,
    samplingRate: response.sampling_rate
  };
}

/**
 * Stacktraces options
 */
let StackTracesDisplayOption = exports.StackTracesDisplayOption = /*#__PURE__*/function (StackTracesDisplayOption) {
  StackTracesDisplayOption["StackTraces"] = "stackTraces";
  StackTracesDisplayOption["Percentage"] = "percentage";
  return StackTracesDisplayOption;
}({});
/**
 * Functions TopN types definition
 */
let TopNType = exports.TopNType = /*#__PURE__*/function (TopNType) {
  TopNType["Containers"] = "containers";
  TopNType["Deployments"] = "deployments";
  TopNType["Executables"] = "executables";
  TopNType["Threads"] = "threads";
  TopNType["Hosts"] = "hosts";
  TopNType["Traces"] = "traces";
  return TopNType;
}({});
/**
 * Get Profiling ES field based on TopN Type
 * @param type TopNType
 * @returns string
 */
function getFieldNameForTopNType(type) {
  return {
    [TopNType.Containers]: _elasticsearch.ProfilingESField.ContainerName,
    [TopNType.Deployments]: _elasticsearch.ProfilingESField.OrchestratorResourceName,
    [TopNType.Executables]: _elasticsearch.ProfilingESField.ProcessExecutableName,
    [TopNType.Threads]: _elasticsearch.ProfilingESField.ProcessThreadName,
    [TopNType.Hosts]: _elasticsearch.ProfilingESField.HostID,
    [TopNType.Traces]: _elasticsearch.ProfilingESField.StacktraceID
  }[type];
}