"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.syncIntegrationsOnRemote = exports.getFollowerIndex = void 0;
var _eq = _interopRequireDefault(require("semver/functions/eq"));
var _gte = _interopRequireDefault(require("semver/functions/gte"));
var _lodash = require("lodash");
var _services = require("../../services");
var _errors = require("../../errors");
var _fleet_synced_integrations = require("../../services/setup/fleet_synced_integrations");
var _packages = require("../../services/epm/packages");
var _constants = require("../../constants");
var _install_errors_helpers = require("../../services/epm/packages/install_errors_helpers");
var _custom_assets = require("./custom_assets");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const MAX_RETRY_ATTEMPTS = 5;
const RETRY_BACKOFF_MINUTES = [5, 10, 20, 40, 60];
const getFollowerIndex = async (esClient, abortController) => {
  const indices = await esClient.indices.get({
    index: _fleet_synced_integrations.FLEET_SYNCED_INTEGRATIONS_CCR_INDEX_PREFIX,
    expand_wildcards: 'all'
  }, {
    signal: abortController.signal
  });
  const indexNames = Object.keys(indices);
  if (indexNames.length > 1) {
    throw new _errors.FleetError(`Not supported to sync multiple indices with prefix ${_fleet_synced_integrations.FLEET_SYNCED_INTEGRATIONS_CCR_INDEX_PREFIX}`);
  }
  if (indexNames.length === 0) {
    return undefined;
  }
  return indexNames[0];
};
exports.getFollowerIndex = getFollowerIndex;
const getSyncedIntegrationsCCRDoc = async (esClient, abortController, logger) => {
  const index = await getFollowerIndex(esClient, abortController);
  const response = await esClient.search({
    index
  }, {
    signal: abortController.signal
  });
  if (response.hits.hits.length === 0) {
    logger.warn(`getSyncedIntegrationsCCRDoc - Sync integration doc not found`);
    return undefined;
  }
  return response.hits.hits[0]._source;
};
async function getSyncIntegrationsEnabled(soClient, remoteEsHosts) {
  const outputs = await _services.outputService.list(soClient);
  const esHosts = outputs.items.filter(output => output.type === 'elasticsearch').flatMap(output => output.hosts);
  const isSyncIntegrationsEnabled = remoteEsHosts === null || remoteEsHosts === void 0 ? void 0 : remoteEsHosts.some(remoteEsHost => {
    return remoteEsHost.sync_integrations && remoteEsHost.hosts.some(host => esHosts.includes(host));
  });
  return isSyncIntegrationsEnabled !== null && isSyncIntegrationsEnabled !== void 0 ? isSyncIntegrationsEnabled : false;
}
async function installPackageIfNotInstalled(savedObjectsClient, pkg, packageClient, logger, abortController) {
  const installation = await packageClient.getInstallation(pkg.package_name);
  if ((installation === null || installation === void 0 ? void 0 : installation.install_status) === 'installed' && (0, _gte.default)(installation.version, pkg.package_version)) {
    logger.debug(`installPackageIfNotInstalled - ${pkg.package_name} already installed`);
    return;
  }
  if ((installation === null || installation === void 0 ? void 0 : installation.install_status) === 'installing') {
    logger.debug(`installPackageIfNotInstalled - ${pkg.package_name} status installing`);
    return;
  }
  if ((installation === null || installation === void 0 ? void 0 : installation.install_status) === 'install_failed') {
    var _installation$latest_, _installation$latest_2, _installation$latest_3;
    const attempt = (_installation$latest_ = (_installation$latest_2 = installation.latest_install_failed_attempts) === null || _installation$latest_2 === void 0 ? void 0 : _installation$latest_2.length) !== null && _installation$latest_ !== void 0 ? _installation$latest_ : 0;
    if (attempt >= MAX_RETRY_ATTEMPTS) {
      logger.debug(`installPackageIfNotInstalled - too many retry attempts at installing ${pkg.package_name}`);
      return;
    }
    const lastRetryAttemptTime = (_installation$latest_3 = installation.latest_install_failed_attempts) === null || _installation$latest_3 === void 0 ? void 0 : _installation$latest_3[0].created_at;
    // retry install if backoff time has passed since the last attempt
    // excluding custom and upload packages from retries
    const shouldRetryInstall = attempt > 0 && lastRetryAttemptTime && Date.now() - Date.parse(lastRetryAttemptTime) > RETRY_BACKOFF_MINUTES[attempt - 1] * 60 * 1000 && (pkg.install_source === 'registry' || pkg.install_source === 'bundled');
    if (!shouldRetryInstall) {
      logger.debug(`installPackageIfNotInstalled - Max retry attempts reached`);
      return;
    }
  }
  try {
    const installResult = await packageClient.installPackage({
      pkgName: pkg.package_name,
      pkgVersion: pkg.package_version,
      keepFailedInstallation: true,
      // using force flag because the package version might not be the latest on this cluster
      force: true
    });
    if (installResult.status === 'installed') {
      logger.info(`Package ${pkg.package_name} installed with version ${pkg.package_version}`);
    }
    if (installResult.error instanceof _errors.PackageNotFoundError) {
      if (abortController.signal.aborted) {
        throw new Error('Task was aborted');
      }
      logger.warn(`Package ${pkg.package_name} with version ${pkg.package_version} not found, trying to install latest version`);
      const installLatestResult = await packageClient.installPackage({
        pkgName: pkg.package_name,
        keepFailedInstallation: true,
        force: true
      });
      if (installLatestResult.status === 'installed') {
        logger.info(`Package ${pkg.package_name} installed with version ${pkg.package_version}`);
      }
    }
  } catch (error) {
    logger.error(`Failed to install package ${pkg.package_name} with version ${pkg.package_version}, error: ${error}`);
    if (error instanceof _errors.PackageNotFoundError && error.message.includes('not found in registry')) {
      await (0, _install_errors_helpers.createOrUpdateFailedInstallStatus)({
        logger,
        savedObjectsClient,
        pkgName: pkg.package_name,
        pkgVersion: pkg.package_version,
        error,
        installSource: pkg === null || pkg === void 0 ? void 0 : pkg.install_source
      });
    }
  }
}
async function uninstallPackageIfInstalled(esClient, savedObjectsClient, pkg, logger) {
  const installation = await (0, _packages.getInstallation)({
    savedObjectsClient,
    pkgName: pkg.package_name
  });
  if (!installation) {
    logger.warn(`uninstallPackageIfInstalled - Installation for ${pkg.package_name} not found`);
    return;
  }
  if (!(installation.install_status === 'installed' && (0, _eq.default)(installation.version, pkg.package_version))) {
    logger.warn(`uninstallPackageIfInstalled - Package ${pkg.package_name} cannot be uninstalled - Found status: ${installation.install_status}, version: ${installation.version} `);
    return;
  }
  try {
    await (0, _packages.removeInstallation)({
      savedObjectsClient,
      pkgName: pkg.package_name,
      pkgVersion: pkg.package_version,
      esClient,
      force: false
    });
    logger.info(`Package ${pkg.package_name} with version ${pkg.package_version} uninstalled via integration syncing`);
  } catch (error) {
    logger.error(`Failed to uninstall package ${pkg.package_name} with version ${pkg.package_version} via integration syncing: ${error.message}`);
  }
}
const syncIntegrationsOnRemote = async (esClient, soClient, packageClient, abortController, logger) => {
  var _syncIntegrationsDoc$, _syncIntegrationsDoc$2;
  const syncIntegrationsDoc = await getSyncedIntegrationsCCRDoc(esClient, abortController, logger);
  const isSyncIntegrationsEnabled = await getSyncIntegrationsEnabled(soClient, syncIntegrationsDoc === null || syncIntegrationsDoc === void 0 ? void 0 : syncIntegrationsDoc.remote_es_hosts);
  if (!isSyncIntegrationsEnabled) {
    logger.debug(`Sync integration not enabled because of remote outputs configuration`);
    return;
  }
  const installedIntegrations = (_syncIntegrationsDoc$ = syncIntegrationsDoc === null || syncIntegrationsDoc === void 0 ? void 0 : syncIntegrationsDoc.integrations.filter(integration => integration.install_status !== 'not_installed')) !== null && _syncIntegrationsDoc$ !== void 0 ? _syncIntegrationsDoc$ : [];
  for (const pkg of installedIntegrations) {
    if (abortController.signal.aborted) {
      throw new Error('Task was aborted');
    }
    await installPackageIfNotInstalled(soClient, pkg, packageClient, logger, abortController);
  }
  const uninstalledIntegrations = (_syncIntegrationsDoc$2 = syncIntegrationsDoc === null || syncIntegrationsDoc === void 0 ? void 0 : syncIntegrationsDoc.integrations.filter(integration => integration.install_status === 'not_installed')) !== null && _syncIntegrationsDoc$2 !== void 0 ? _syncIntegrationsDoc$2 : [];
  for (const pkg of uninstalledIntegrations) {
    if (abortController.signal.aborted) {
      throw new Error('Task was aborted');
    }
    await uninstallPackageIfInstalled(esClient, soClient, pkg, logger);
  }
  await clearCustomAssetFailedAttempts(soClient, syncIntegrationsDoc);
  for (const customAsset of Object.values((_syncIntegrationsDoc$3 = syncIntegrationsDoc === null || syncIntegrationsDoc === void 0 ? void 0 : syncIntegrationsDoc.custom_assets) !== null && _syncIntegrationsDoc$3 !== void 0 ? _syncIntegrationsDoc$3 : {})) {
    var _syncIntegrationsDoc$3;
    if (abortController.signal.aborted) {
      throw new Error('Task was aborted');
    }
    try {
      await (0, _custom_assets.installCustomAsset)(customAsset, esClient, abortController, logger);
    } catch (error) {
      logger.error(`Failed to install ${customAsset.type} ${customAsset.name}, error: ${error}`);
      await updateCustomAssetFailedAttempts(soClient, customAsset, error, logger);
    }
  }
};
exports.syncIntegrationsOnRemote = syncIntegrationsOnRemote;
async function clearCustomAssetFailedAttempts(soClient, syncIntegrationsDoc) {
  var _syncIntegrationsDoc$4;
  const customAssetPackages = (0, _lodash.uniq)(Object.values((_syncIntegrationsDoc$4 = syncIntegrationsDoc === null || syncIntegrationsDoc === void 0 ? void 0 : syncIntegrationsDoc.custom_assets) !== null && _syncIntegrationsDoc$4 !== void 0 ? _syncIntegrationsDoc$4 : {}).map(customAsset => {
    return customAsset.package_name;
  }));
  for (const pkgName of customAssetPackages) {
    await soClient.update(_constants.PACKAGES_SAVED_OBJECT_TYPE, pkgName, {
      latest_custom_asset_install_failed_attempts: {}
    });
  }
}
async function updateCustomAssetFailedAttempts(savedObjectsClient, customAsset, error, logger) {
  try {
    await savedObjectsClient.update(_constants.PACKAGES_SAVED_OBJECT_TYPE, customAsset.package_name, {
      latest_custom_asset_install_failed_attempts: {
        [`${customAsset.type}:${customAsset.name}`]: {
          type: customAsset.type,
          name: customAsset.name,
          error: {
            name: error.name,
            message: error.message,
            stack: error.stack
          },
          created_at: new Date().toISOString()
        }
      }
    });
  } catch (err) {
    logger.warn(`Error occurred while updating custom asset failed attempts: ${err}`);
  }
}