"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getSchedule = getSchedule;
var _server = require("@kbn/task-manager-plugin/server");
var _result_type = require("../../lib/result_type");
var _is_alerting_error = require("../../lib/is_alerting_error");
var _common = require("../../../common");
var _error_with_type = require("../../lib/error_with_type");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const CONNECTIVITY_RETRY_INTERVAL = '5m';
const CLUSTER_BLOCKED_EXCEPTION_RETRY_INTERVAL = '1m';
function getSchedule({
  schedule,
  ruleId,
  spaceId,
  retryInterval,
  logger
}) {
  return (0, _result_type.resolveErr)(schedule, error => {
    if ((0, _is_alerting_error.isAlertSavedObjectNotFoundError)(error, ruleId)) {
      const spaceMessage = spaceId ? `in the "${spaceId}" space ` : '';
      logger.warn(`Unable to execute rule "${ruleId}" ${spaceMessage}because ${error.message} - this rule will not be rescheduled. To restart rule execution, try disabling and re-enabling this rule.`);
      (0, _server.throwUnrecoverableError)(error);
    }

    // Set retry interval smaller for ES connectivity errors
    if ((0, _is_alerting_error.isEsUnavailableError)(error, ruleId)) {
      retryInterval = (0, _common.parseDuration)(retryInterval) > (0, _common.parseDuration)(CONNECTIVITY_RETRY_INTERVAL) ? CONNECTIVITY_RETRY_INTERVAL : retryInterval;
    }
    if ((0, _error_with_type.isClusterBlockError)(error)) {
      retryInterval = CLUSTER_BLOCKED_EXCEPTION_RETRY_INTERVAL;
    }
    return {
      interval: retryInterval
    };
  });
}