"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.RiskScoreDataClient = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _server = require("@kbn/alerting-plugin/server");
var _common = require("@kbn/alerting-plugin/common");
var _configurations = require("./configurations");
var _create_datastream = require("../utils/create_datastream");
var _risk_engine_data_writer = require("./risk_engine_data_writer");
var _risk_engine = require("../../../../common/entity_analytics/risk_engine");
var _transforms = require("../utils/transforms");
var _get_risk_inputs_index = require("./get_risk_inputs_index");
var _create_or_update_index = require("../utils/create_or_update_index");
var _retry_transient_es_errors = require("../utils/retry_transient_es_errors");
var _audit = require("./audit");
var _audit2 = require("../audit");
var _event_ingested_pipeline = require("../utils/event_ingested_pipeline");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class RiskScoreDataClient {
  constructor(options) {
    (0, _defineProperty2.default)(this, "writerCache", new Map());
    (0, _defineProperty2.default)(this, "refreshRiskScoreIndex", async () => {
      await this.options.esClient.indices.refresh({
        index: (0, _risk_engine.getRiskScoreTimeSeriesIndex)(this.options.namespace)
      });
    });
    (0, _defineProperty2.default)(this, "getRiskInputsIndex", ({
      dataViewId
    }) => (0, _get_risk_inputs_index.getRiskInputsIndex)({
      dataViewId,
      logger: this.options.logger,
      soClient: this.options.soClient
    }));
    (0, _defineProperty2.default)(this, "createOrUpdateRiskScoreLatestIndex", async () => {
      await (0, _create_or_update_index.createOrUpdateIndex)({
        esClient: this.options.esClient,
        logger: this.options.logger,
        options: {
          index: (0, _risk_engine.getRiskScoreLatestIndex)(this.options.namespace),
          mappings: (0, _common.mappingFromFieldMap)(_configurations.riskScoreFieldMap, false),
          settings: {
            // set to null because it was previously set but we now want it to be removed
            'index.default_pipeline': null
          }
        }
      });
    });
    (0, _defineProperty2.default)(this, "createOrUpdateRiskScoreComponentTemplate", async () => (0, _server.createOrUpdateComponentTemplate)({
      logger: this.options.logger,
      esClient: this.options.esClient,
      template: {
        name: (0, _configurations.nameSpaceAwareMappingsComponentName)(this.options.namespace),
        _meta: {
          managed: true
        },
        template: {
          settings: {},
          mappings: (0, _common.mappingFromFieldMap)(_configurations.riskScoreFieldMap, 'strict')
        }
      },
      totalFieldsLimit: _configurations.totalFieldsLimit
    }));
    (0, _defineProperty2.default)(this, "createOrUpdateRiskScoreIndexTemplate", async () => {
      const indexPatterns = (0, _configurations.getIndexPatternDataStream)(this.options.namespace);
      const indexMetadata = {
        kibana: {
          version: this.options.kibanaVersion
        },
        managed: true,
        namespace: this.options.namespace
      };
      return (0, _server.createOrUpdateIndexTemplate)({
        logger: this.options.logger,
        esClient: this.options.esClient,
        template: {
          name: indexPatterns.template,
          data_stream: {
            hidden: true
          },
          index_patterns: [indexPatterns.alias],
          composed_of: [(0, _configurations.nameSpaceAwareMappingsComponentName)(this.options.namespace)],
          template: {
            lifecycle: {},
            settings: {
              'index.mapping.total_fields.limit': _configurations.totalFieldsLimit,
              'index.default_pipeline': (0, _event_ingested_pipeline.getIngestPipelineName)(this.options.namespace)
            },
            mappings: {
              dynamic: false,
              _meta: indexMetadata
            }
          },
          _meta: indexMetadata
        }
      });
    });
    (0, _defineProperty2.default)(this, "rolloverRiskScoreTimeSeriesIndex", async () => (0, _create_datastream.rolloverDataStream)({
      esClient: this.options.esClient,
      logger: this.options.logger,
      dataStreamName: (0, _risk_engine.getRiskScoreTimeSeriesIndex)(this.options.namespace)
    }));
    (0, _defineProperty2.default)(this, "copyTimestampToEventIngestedForRiskScore", abortSignal => {
      return this.options.esClient.updateByQuery({
        index: (0, _risk_engine.getRiskScoreLatestIndex)(this.options.namespace),
        conflicts: 'proceed',
        ignore_unavailable: true,
        allow_no_indices: true,
        query: {
          bool: {
            must_not: {
              exists: {
                field: 'event.ingested'
              }
            }
          }
        },
        script: {
          source: 'ctx._source.event.ingested = ctx._source.@timestamp',
          lang: 'painless'
        }
      }, {
        requestTimeout: '5m',
        retryOnTimeout: true,
        maxRetries: 2,
        signal: abortSignal
      });
    });
    this.options = options;
  }
  async getWriter({
    namespace
  }) {
    if (this.writerCache.get(namespace)) {
      return this.writerCache.get(namespace);
    }
    const indexPatterns = (0, _configurations.getIndexPatternDataStream)(namespace);
    await this.initializeWriter(namespace, indexPatterns.alias);
    return this.writerCache.get(namespace);
  }
  async initializeWriter(namespace, index) {
    const writer = new _risk_engine_data_writer.RiskEngineDataWriter({
      esClient: this.options.esClient,
      namespace,
      index,
      logger: this.options.logger
    });
    this.writerCache.set(namespace, writer);
    return writer;
  }
  async init() {
    const namespace = this.options.namespace;
    const esClient = this.options.esClient;
    try {
      var _this$options$auditLo;
      await (0, _event_ingested_pipeline.createEventIngestedPipeline)(esClient, namespace);
      await this.createOrUpdateRiskScoreComponentTemplate();
      await this.createOrUpdateRiskScoreIndexTemplate();
      const indexPatterns = (0, _configurations.getIndexPatternDataStream)(namespace);
      await (0, _create_datastream.createDataStream)({
        logger: this.options.logger,
        esClient,
        totalFieldsLimit: _configurations.totalFieldsLimit,
        indexPatterns
      });
      await this.createOrUpdateRiskScoreLatestIndex();
      const transformId = (0, _transforms.getLatestTransformId)(namespace);
      await (0, _transforms.createTransform)({
        esClient,
        logger: this.options.logger,
        transform: {
          transform_id: transformId,
          ...(0, _configurations.getTransformOptions)({
            dest: (0, _risk_engine.getRiskScoreLatestIndex)(namespace),
            source: [indexPatterns.alias],
            namespace: this.options.namespace
          })
        }
      });
      (_this$options$auditLo = this.options.auditLogger) === null || _this$options$auditLo === void 0 ? void 0 : _this$options$auditLo.log({
        message: 'System installed risk engine Elasticsearch components',
        event: {
          action: _audit.RiskScoreAuditActions.RISK_ENGINE_INSTALL,
          category: _audit2.AUDIT_CATEGORY.DATABASE,
          type: _audit2.AUDIT_TYPE.CHANGE,
          outcome: _audit2.AUDIT_OUTCOME.SUCCESS
        }
      });
    } catch (error) {
      this.options.logger.error(`Error initializing risk engine resources: ${error.message}`);
      throw error;
    }
  }

  /**
   * Deletes all resources created by init().
   * It returns an array of errors that occurred during the deletion.
   *
   * WARNING: It will remove all data.
   */
  async tearDown() {
    const namespace = this.options.namespace;
    const esClient = this.options.esClient;
    const indexPatterns = (0, _configurations.getIndexPatternDataStream)(namespace);
    const errors = [];
    const addError = e => errors.push(e);
    await (0, _transforms.deleteTransform)({
      esClient,
      logger: this.options.logger,
      transformId: (0, _transforms.getLatestTransformId)(namespace),
      deleteData: true
    }).catch(addError);
    await esClient.indices.deleteDataStream({
      name: indexPatterns.alias
    }, {
      ignore: [404]
    }).catch(addError);
    await esClient.indices.deleteIndexTemplate({
      name: indexPatterns.template
    }, {
      ignore: [404]
    }).catch(addError);
    await esClient.cluster.deleteComponentTemplate({
      name: (0, _configurations.nameSpaceAwareMappingsComponentName)(namespace)
    }, {
      ignore: [404]
    }).catch(addError);
    await esClient.cluster.deleteComponentTemplate({
      name: _configurations.mappingComponentName
    }, {
      ignore: [404]
    }).catch(addError);
    return errors;
  }

  /**
   * Ensures that configuration migrations for risk score indices are seamlessly handled across Kibana upgrades.
   *
   * Upgrades:
   * - Migrating to 8.12+ requires a change to the risk score latest transform index's 'dynamic' setting to ensure that
   * unmapped fields are allowed within stored documents.
   *
   */
  async upgradeIfNeeded() {
    const desiredDynamicValue = 'false';
    const currentDynamicValue = await this.getRiskScoreLatestIndexDynamicConfiguration();
    if (currentDynamicValue !== desiredDynamicValue) {
      await this.setRiskScoreLatestIndexDynamicConfiguration(desiredDynamicValue);
    }
  }
  async getRiskScoreLatestIndexDynamicConfiguration() {
    var _riskScoreLatestIndex, _riskScoreLatestIndex2, _riskScoreLatestIndex3;
    const riskScoreLatestIndexName = (0, _risk_engine.getRiskScoreLatestIndex)(this.options.namespace);
    const riskScoreLatestIndexResponse = await (0, _retry_transient_es_errors.retryTransientEsErrors)(() => this.options.esClient.indices.get({
      index: riskScoreLatestIndexName
    }), {
      logger: this.options.logger
    });
    return (_riskScoreLatestIndex = riskScoreLatestIndexResponse[riskScoreLatestIndexName]) === null || _riskScoreLatestIndex === void 0 ? void 0 : (_riskScoreLatestIndex2 = _riskScoreLatestIndex.mappings) === null || _riskScoreLatestIndex2 === void 0 ? void 0 : (_riskScoreLatestIndex3 = _riskScoreLatestIndex2.dynamic) === null || _riskScoreLatestIndex3 === void 0 ? void 0 : _riskScoreLatestIndex3.toString();
  }

  /**
   * Sets the risk score latest index's 'dynamic' mapping property to the desired value.
   * @throws Error if the index does not exist.
   */
  async setRiskScoreLatestIndexDynamicConfiguration(dynamic) {
    return (0, _retry_transient_es_errors.retryTransientEsErrors)(() => this.options.esClient.indices.putMapping({
      index: (0, _risk_engine.getRiskScoreLatestIndex)(this.options.namespace),
      dynamic
    }), {
      logger: this.options.logger
    });
  }
  async reinstallTransform() {
    const esClient = this.options.esClient;
    const namespace = this.options.namespace;
    const transformId = (0, _transforms.getLatestTransformId)(namespace);
    const indexPatterns = (0, _configurations.getIndexPatternDataStream)(namespace);
    await (0, _transforms.stopTransform)({
      esClient,
      logger: this.options.logger,
      transformId
    });
    await (0, _transforms.deleteTransform)({
      esClient,
      logger: this.options.logger,
      transformId
    });
    await (0, _transforms.createTransform)({
      esClient,
      logger: this.options.logger,
      transform: {
        transform_id: transformId,
        ...(0, _configurations.getTransformOptions)({
          dest: (0, _risk_engine.getRiskScoreLatestIndex)(namespace),
          source: [indexPatterns.alias],
          namespace: this.options.namespace
        })
      }
    });
  }
}
exports.RiskScoreDataClient = RiskScoreDataClient;