"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.buildParseSearchQuery = void 0;
var _rxjs = require("rxjs");
var _eui = require("@elastic/eui");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const buildParseSearchQuery = ({
  cache
}) => {
  return async (query, {
    tagField = 'tag',
    useName = true
  } = {}) => {
    let parsed;
    try {
      parsed = _eui.Query.parse(query);
    } catch (e) {
      return {
        searchTerm: query,
        tagReferences: [],
        tagReferencesToExclude: [],
        valid: false
      };
    }

    // from other usages of `Query.parse` in the codebase, it seems that
    // for empty term, the parsed query can be undefined, even if the type def state otherwise.
    if (!query) {
      return {
        searchTerm: '',
        tagReferences: [],
        tagReferencesToExclude: [],
        valid: true
      };
    }
    let searchTerm = '';
    if (parsed.ast.getTermClauses().length) {
      searchTerm = parsed.ast.getTermClauses().map(clause => clause.value).join(' ');
    }
    let tagReferences = [];
    let tagReferencesToExclude = [];
    if (parsed.ast.getFieldClauses(tagField)) {
      // The query can have clauses that either *must* match or *must_not* match
      // We will retrieve the list of name for both list and convert them to references
      const {
        selectedTags,
        excludedTags
      } = parsed.ast.getFieldClauses(tagField).reduce((acc, clause) => {
        if (clause.match === 'must') {
          acc.selectedTags = clause.value;
        } else if (clause.match === 'must_not') {
          acc.excludedTags = clause.value;
        }
        return acc;
      }, {
        selectedTags: [],
        excludedTags: []
      });
      const tagsInCache = await (0, _rxjs.lastValueFrom)(cache.getState$({
        waitForInitialization: true
      }).pipe((0, _rxjs.first)()));
      const tagsToReferences = tagNames => {
        if (useName) {
          const references = [];
          tagNames.forEach(tagName => {
            const found = tagsInCache.find(tag => tag.name === tagName);
            if (found) {
              references.push({
                type: 'tag',
                id: found.id
              });
            }
          });
          return references;
        } else {
          return tagNames.map(tagId => ({
            type: 'tag',
            id: tagId
          }));
        }
      };
      tagReferences = tagsToReferences(selectedTags);
      tagReferencesToExclude = tagsToReferences(excludedTags);
    }
    return {
      searchTerm,
      tagReferences,
      tagReferencesToExclude,
      valid: true
    };
  };
};
exports.buildParseSearchQuery = buildParseSearchQuery;