"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getCurrentTitle = getCurrentTitle;
exports.isLegacyEditorEmbeddable = isLegacyEditorEmbeddable;
exports.setBreadcrumbsTitle = setBreadcrumbsTitle;
exports.useNavigateBackToApp = useNavigateBackToApp;
exports.useShortUrlService = useShortUrlService;
var _i18n = require("@kbn/i18n");
var _react = require("react");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const VISUALIZE_APP_ID = 'visualize';
function isLegacyEditorEmbeddable(initialContext) {
  return Boolean(initialContext && 'isEmbeddable' in initialContext && initialContext.isEmbeddable);
}
function getCurrentTitle(persistedDoc, isByValueMode, initialContext) {
  if (persistedDoc) {
    if (isByValueMode) {
      return _i18n.i18n.translate('xpack.lens.breadcrumbsByValue', {
        defaultMessage: 'Edit visualization'
      });
    }
    if (persistedDoc.title) {
      return persistedDoc.title;
    }
  }
  if (!(persistedDoc !== null && persistedDoc !== void 0 && persistedDoc.title) && isLegacyEditorEmbeddable(initialContext)) {
    return _i18n.i18n.translate('xpack.lens.breadcrumbsEditInLensFromDashboard', {
      defaultMessage: 'Converting {title} visualization',
      values: {
        title: initialContext.title ? `"${initialContext.title}"` : initialContext.visTypeTitle
      }
    });
  }
  return _i18n.i18n.translate('xpack.lens.breadcrumbsCreate', {
    defaultMessage: 'Create'
  });
}
function setBreadcrumbsTitle({
  application,
  serverless,
  chrome
}, {
  isByValueMode,
  originatingAppName,
  redirectToOrigin,
  isFromLegacyEditor,
  currentDocTitle
}) {
  const breadcrumbs = [];
  if (isFromLegacyEditor && originatingAppName && redirectToOrigin) {
    breadcrumbs.push({
      onClick: () => {
        redirectToOrigin();
      },
      text: originatingAppName
    });
  }
  if (!isByValueMode) {
    breadcrumbs.push({
      href: application.getUrlForApp(VISUALIZE_APP_ID),
      onClick: e => {
        application.navigateToApp(VISUALIZE_APP_ID, {
          path: '/'
        });
        e.preventDefault();
      },
      text: _i18n.i18n.translate('xpack.lens.breadcrumbsTitle', {
        defaultMessage: 'Visualize library'
      })
    });
  }
  const currentDocBreadcrumb = {
    text: currentDocTitle
  };
  breadcrumbs.push(currentDocBreadcrumb);
  if (serverless !== null && serverless !== void 0 && serverless.setBreadcrumbs) {
    // TODO: https://github.com/elastic/kibana/issues/163488
    // for now, serverless breadcrumbs only set the title,
    // the rest of the breadcrumbs are handled by the serverless navigation
    // the serverless navigation is not yet aware of the byValue/originatingApp context
    serverless.setBreadcrumbs(currentDocBreadcrumb);
  } else {
    chrome.setBreadcrumbs(breadcrumbs);
  }
}
function useShortUrlService(locator, share) {
  const shortUrls = (0, _react.useMemo)(() => share === null || share === void 0 ? void 0 : share.url.shortUrls.get(null), [share]);
  // remember latest URL based on the configuration
  // url_panel_content has a similar logic
  const shareURLCache = (0, _react.useRef)({
    params: '',
    url: ''
  });
  return (0, _react.useCallback)(async params => {
    const cacheKey = JSON.stringify(params);
    if (shareURLCache.current.params === cacheKey) {
      return shareURLCache.current.url;
    }
    if (locator && shortUrls) {
      // This is a stripped down version of what the share URL plugin is doing
      const shortUrl = await shortUrls.createWithLocator({
        locator,
        params
      });
      const absoluteShortUrl = await shortUrl.locator.getUrl(shortUrl.params, {
        absolute: true
      });
      shareURLCache.current = {
        params: cacheKey,
        url: absoluteShortUrl
      };
      return absoluteShortUrl;
    }
    return '';
  }, [locator, shortUrls]);
}
function useNavigateBackToApp({
  application,
  onAppLeave,
  legacyEditorAppName,
  legacyEditorAppUrl,
  initialDocFromContext,
  persistedDoc,
  isLensEqual
}) {
  const [shouldShowGoBackToVizEditorModal, setIsGoBackToVizEditorModalVisible] = (0, _react.useState)(false);
  /** Shared logic to navigate back to the originating viz editor app */
  const navigateBackToVizEditor = (0, _react.useCallback)(() => {
    if (legacyEditorAppUrl) {
      onAppLeave(actions => {
        return actions.default();
      });
      application.navigateToApp(legacyEditorAppName || VISUALIZE_APP_ID, {
        path: legacyEditorAppUrl
      });
    }
  }, [application, legacyEditorAppName, legacyEditorAppUrl, onAppLeave]);

  // if users comes to Lens from the Viz editor, they should have the option to navigate back
  // used for TopNavMenu
  const goBackToOriginatingApp = (0, _react.useCallback)(() => {
    if (legacyEditorAppUrl) {
      if ([initialDocFromContext, persistedDoc].some(isLensEqual)) {
        navigateBackToVizEditor();
      } else {
        setIsGoBackToVizEditorModalVisible(true);
      }
    }
  }, [legacyEditorAppUrl, initialDocFromContext, persistedDoc, isLensEqual, navigateBackToVizEditor, setIsGoBackToVizEditorModalVisible]);

  // Used for Saving Modal
  const navigateToVizEditor = (0, _react.useCallback)(() => {
    setIsGoBackToVizEditorModalVisible(false);
    navigateBackToVizEditor();
  }, [navigateBackToVizEditor, setIsGoBackToVizEditorModalVisible]);
  return {
    shouldShowGoBackToVizEditorModal,
    goBackToOriginatingApp,
    navigateToVizEditor,
    closeGoBackToVizEditorModal: () => setIsGoBackToVizEditorModalVisible(false)
  };
}