"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.updateAgentHandler = exports.postRetrieveAgentsByActionsHandler = exports.postBulkAgentReassignHandler = exports.postAgentReassignHandler = exports.getAvailableVersionsHandler = exports.getAgentsHandler = exports.getAgentUploadsHandler = exports.getAgentUploadFileHandler = exports.getAgentTagsHandler = exports.getAgentStatusRuntimeFieldHandler = exports.getAgentStatusForAgentPolicyHandler = exports.getAgentHandler = exports.getAgentDataHandler = exports.getActionStatusHandler = exports.deleteAgentUploadFileHandler = exports.deleteAgentHandler = exports.bulkUpdateAgentTagsHandler = void 0;
var _lodash = require("lodash");
var _server = require("@kbn/core/server");
var _errors = require("../../errors");
var _agents = _interopRequireWildcard(require("../../services/agents"));
var AgentService = _agents;
var _agent_metrics = require("../../services/agents/agent_metrics");
var _agent_namespaces = require("../../services/spaces/agent_namespaces");
var _get_current_namespace = require("../../services/spaces/get_current_namespace");
var _packages = require("../../services/epm/packages");
var _template = require("../../services/epm/elasticsearch/template/template");
var _build_status_runtime_field = require("../../services/agents/build_status_runtime_field");
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && {}.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

async function verifyNamespace(agent, namespace) {
  if (!(await (0, _agent_namespaces.isAgentInNamespace)(agent, namespace))) {
    throw new _errors.FleetNotFoundError(`${agent.id} not found in namespace`);
  }
}
const getAgentHandler = async (context, request, response) => {
  try {
    const [coreContext, fleetContext] = await Promise.all([context.core, context.fleet]);
    const esClientCurrentUser = coreContext.elasticsearch.client.asCurrentUser;
    let agent = await fleetContext.agentClient.asCurrentUser.getAgent(request.params.agentId);
    await verifyNamespace(agent, (0, _get_current_namespace.getCurrentNamespace)(coreContext.savedObjects.client));
    if (request.query.withMetrics) {
      agent = (await (0, _agent_metrics.fetchAndAssignAgentMetrics)(esClientCurrentUser, [agent]))[0];
    }
    const body = {
      item: agent
    };
    return response.ok({
      body
    });
  } catch (error) {
    if (_server.SavedObjectsErrorHelpers.isNotFoundError(error)) {
      return response.notFound({
        body: {
          message: `Agent ${request.params.agentId} not found`
        }
      });
    }
    throw error;
  }
};
exports.getAgentHandler = getAgentHandler;
const deleteAgentHandler = async (context, request, response) => {
  const [coreContext, fleetContext] = await Promise.all([context.core, context.fleet]);
  const esClient = coreContext.elasticsearch.client.asInternalUser;
  try {
    const agent = await fleetContext.agentClient.asCurrentUser.getAgent(request.params.agentId);
    await verifyNamespace(agent, (0, _get_current_namespace.getCurrentNamespace)(coreContext.savedObjects.client));
    await AgentService.deleteAgent(esClient, request.params.agentId);
    const body = {
      action: 'deleted'
    };
    return response.ok({
      body
    });
  } catch (error) {
    if (error.isBoom) {
      return response.customError({
        statusCode: error.output.statusCode,
        body: {
          message: `Agent ${request.params.agentId} not found`
        }
      });
    }
    throw error;
  }
};
exports.deleteAgentHandler = deleteAgentHandler;
const updateAgentHandler = async (context, request, response) => {
  const [coreContext, fleetContext] = await Promise.all([context.core, context.fleet]);
  const esClient = coreContext.elasticsearch.client.asInternalUser;
  const soClient = coreContext.savedObjects.client;
  const partialAgent = {};
  if (request.body.user_provided_metadata) {
    partialAgent.user_provided_metadata = request.body.user_provided_metadata;
  }
  if (request.body.tags) {
    partialAgent.tags = (0, _lodash.uniq)(request.body.tags);
  }
  try {
    const agent = await fleetContext.agentClient.asCurrentUser.getAgent(request.params.agentId);
    await verifyNamespace(agent, (0, _get_current_namespace.getCurrentNamespace)(soClient));
    await AgentService.updateAgent(esClient, request.params.agentId, partialAgent);
    const body = {
      item: await AgentService.getAgentById(esClient, soClient, request.params.agentId)
    };
    return response.ok({
      body
    });
  } catch (error) {
    if (error.isBoom && error.output.statusCode === 404) {
      return response.notFound({
        body: {
          message: `Agent ${request.params.agentId} not found`
        }
      });
    }
    throw error;
  }
};
exports.updateAgentHandler = updateAgentHandler;
const bulkUpdateAgentTagsHandler = async (context, request, response) => {
  var _request$body$tagsToA, _request$body$tagsToR;
  const coreContext = await context.core;
  const esClient = coreContext.elasticsearch.client.asInternalUser;
  const soClient = coreContext.savedObjects.client;
  const agentOptions = Array.isArray(request.body.agents) ? {
    agentIds: request.body.agents
  } : {
    kuery: request.body.agents,
    showInactive: request.body.includeInactive
  };
  const results = await AgentService.updateAgentTags(soClient, esClient, {
    ...agentOptions,
    batchSize: request.body.batchSize
  }, (_request$body$tagsToA = request.body.tagsToAdd) !== null && _request$body$tagsToA !== void 0 ? _request$body$tagsToA : [], (_request$body$tagsToR = request.body.tagsToRemove) !== null && _request$body$tagsToR !== void 0 ? _request$body$tagsToR : []);
  return response.ok({
    body: {
      actionId: results.actionId
    }
  });
};
exports.bulkUpdateAgentTagsHandler = bulkUpdateAgentTagsHandler;
const getAgentsHandler = async (context, request, response) => {
  const [coreContext, fleetContext] = await Promise.all([context.core, context.fleet]);
  const {
    agentClient
  } = fleetContext;
  const esClientCurrentUser = coreContext.elasticsearch.client.asCurrentUser;

  // Unwrap searchAfter from request query
  let searchAfter;
  if (request.query.searchAfter) {
    try {
      const searchAfterArray = JSON.parse(request.query.searchAfter);
      if (!Array.isArray(searchAfterArray) || searchAfterArray.length === 0) {
        response.badRequest({
          body: {
            message: 'searchAfter must be a non-empty array'
          }
        });
      } else {
        searchAfter = searchAfterArray;
      }
    } catch (e) {
      response.badRequest({
        body: {
          message: 'searchAfter must be a non-empty array'
        }
      });
    }
  }
  const agentRes = await agentClient.asCurrentUser.listAgents({
    page: request.query.page,
    perPage: request.query.perPage,
    showAgentless: request.query.showAgentless,
    showInactive: request.query.showInactive,
    showUpgradeable: request.query.showUpgradeable,
    kuery: request.query.kuery,
    sortField: request.query.sortField,
    sortOrder: request.query.sortOrder,
    searchAfter,
    openPit: request.query.openPit,
    pitId: request.query.pitId,
    pitKeepAlive: request.query.pitKeepAlive,
    getStatusSummary: request.query.getStatusSummary
  });
  const {
    total,
    page,
    perPage,
    statusSummary,
    pit
  } = agentRes;
  let {
    agents
  } = agentRes;

  // Assign metrics
  if (request.query.withMetrics) {
    agents = await (0, _agent_metrics.fetchAndAssignAgentMetrics)(esClientCurrentUser, agents);
  }

  // Retrieve last agent to use for nextSearchAfter
  const lastAgent = agents.length > 0 ? agents[agents.length - 1] : undefined;
  const body = {
    items: agents,
    total,
    page,
    perPage,
    ...(lastAgent && lastAgent.sort ? {
      nextSearchAfter: JSON.stringify(lastAgent.sort)
    } : {}),
    ...(pit ? {
      pit
    } : {}),
    ...(statusSummary ? {
      statusSummary
    } : {})
  };
  return response.ok({
    body
  });
};
exports.getAgentsHandler = getAgentsHandler;
const getAgentTagsHandler = async (context, request, response) => {
  const coreContext = await context.core;
  const esClient = coreContext.elasticsearch.client.asInternalUser;
  const soClient = coreContext.savedObjects.client;
  const tags = await AgentService.getAgentTags(soClient, esClient, {
    showInactive: request.query.showInactive,
    kuery: request.query.kuery
  });
  const body = {
    items: tags
  };
  return response.ok({
    body
  });
};
exports.getAgentTagsHandler = getAgentTagsHandler;
const postAgentReassignHandler = async (context, request, response) => {
  const coreContext = await context.core;
  const soClient = coreContext.savedObjects.client;
  const esClient = coreContext.elasticsearch.client.asInternalUser;
  await AgentService.reassignAgent(soClient, esClient, request.params.agentId, request.body.policy_id);
  const body = {};
  return response.ok({
    body
  });
};
exports.postAgentReassignHandler = postAgentReassignHandler;
const postBulkAgentReassignHandler = async (context, request, response) => {
  const coreContext = await context.core;
  const soClient = coreContext.savedObjects.client;
  const esClient = coreContext.elasticsearch.client.asInternalUser;
  const agentOptions = Array.isArray(request.body.agents) ? {
    agentIds: request.body.agents
  } : {
    kuery: request.body.agents,
    showInactive: request.body.includeInactive
  };
  const results = await AgentService.reassignAgents(soClient, esClient, {
    ...agentOptions,
    batchSize: request.body.batchSize
  }, request.body.policy_id);
  return response.ok({
    body: {
      actionId: results.actionId
    }
  });
};
exports.postBulkAgentReassignHandler = postBulkAgentReassignHandler;
const getAgentStatusForAgentPolicyHandler = async (context, request, response) => {
  const [coreContext, fleetContext] = await Promise.all([context.core, context.fleet]);
  const esClient = coreContext.elasticsearch.client.asInternalUser;
  const soClient = fleetContext.internalSoClient;
  const parsePolicyIds = policyIds => {
    if (!policyIds || !policyIds.length) {
      return undefined;
    }
    return Array.isArray(policyIds) ? policyIds : [policyIds];
  };
  const results = await (0, _agents.getAgentStatusForAgentPolicy)(esClient, soClient, request.query.policyId, request.query.kuery, coreContext.savedObjects.client.getCurrentNamespace(), parsePolicyIds(request.query.policyIds));
  const body = {
    results: (0, _lodash.omit)(results, 'total')
  };
  return response.ok({
    body
  });
};
exports.getAgentStatusForAgentPolicyHandler = getAgentStatusForAgentPolicyHandler;
const getAgentDataHandler = async (context, request, response) => {
  const coreContext = await context.core;
  const esClient = coreContext.elasticsearch.client.asCurrentUser;
  const agentsIds = isStringArray(request.query.agentsIds) ? request.query.agentsIds : [request.query.agentsIds];
  const {
    pkgName,
    pkgVersion,
    previewData: returnDataPreview
  } = request.query;

  // If a package is specified, get data stream patterns for that package
  // and scope incoming data query to that pattern
  let dataStreamPattern;
  if (pkgName && pkgVersion) {
    const packageInfo = await (0, _packages.getPackageInfo)({
      savedObjectsClient: coreContext.savedObjects.client,
      prerelease: true,
      pkgName,
      pkgVersion
    });
    dataStreamPattern = (packageInfo.data_streams || []).map(ds => (0, _template.generateTemplateIndexPattern)(ds)).join(',');
  }
  const {
    items,
    dataPreview
  } = await AgentService.getIncomingDataByAgentsId({
    esClient,
    agentsIds,
    dataStreamPattern,
    returnDataPreview
  });
  const body = {
    items,
    dataPreview
  };
  return response.ok({
    body
  });
};
exports.getAgentDataHandler = getAgentDataHandler;
function isStringArray(arr) {
  return Array.isArray(arr) && arr.every(p => typeof p === 'string');
}
const getAgentStatusRuntimeFieldHandler = async (context, request, response) => {
  const runtimeFields = await (0, _build_status_runtime_field.buildAgentStatusRuntimeField)();
  return response.ok({
    body: runtimeFields.status.script.source
  });
};
exports.getAgentStatusRuntimeFieldHandler = getAgentStatusRuntimeFieldHandler;
const getAvailableVersionsHandler = async (context, request, response) => {
  const availableVersions = await AgentService.getAvailableVersions();
  const body = {
    items: availableVersions
  };
  return response.ok({
    body
  });
};
exports.getAvailableVersionsHandler = getAvailableVersionsHandler;
const getActionStatusHandler = async (context, request, response) => {
  const coreContext = await context.core;
  const esClient = coreContext.elasticsearch.client.asInternalUser;
  const actionStatuses = await AgentService.getActionStatuses(esClient, request.query, (0, _get_current_namespace.getCurrentNamespace)(coreContext.savedObjects.client));
  const body = {
    items: actionStatuses
  };
  return response.ok({
    body
  });
};
exports.getActionStatusHandler = getActionStatusHandler;
const postRetrieveAgentsByActionsHandler = async (context, request, response) => {
  const coreContext = await context.core;
  const esClient = coreContext.elasticsearch.client.asInternalUser;
  const agents = await AgentService.getAgentsByActionsIds(esClient, request.body.actionIds);
  const body = {
    items: agents
  };
  return response.ok({
    body
  });
};
exports.postRetrieveAgentsByActionsHandler = postRetrieveAgentsByActionsHandler;
const getAgentUploadsHandler = async (context, request, response) => {
  const coreContext = await context.core;
  const esClient = coreContext.elasticsearch.client.asInternalUser;
  const body = {
    items: await AgentService.getAgentUploads(esClient, request.params.agentId)
  };
  return response.ok({
    body
  });
};
exports.getAgentUploadsHandler = getAgentUploadsHandler;
const getAgentUploadFileHandler = async (context, request, response) => {
  const coreContext = await context.core;
  const esClient = coreContext.elasticsearch.client.asInternalUser;
  const resp = await AgentService.getAgentUploadFile(esClient, request.params.fileId, request.params.fileName);
  return response.ok(resp);
};
exports.getAgentUploadFileHandler = getAgentUploadFileHandler;
const deleteAgentUploadFileHandler = async (context, request, response) => {
  const coreContext = await context.core;
  const esClient = coreContext.elasticsearch.client.asInternalUser;
  const resp = await AgentService.deleteAgentUploadFile(esClient, request.params.fileId);
  return response.ok({
    body: resp
  });
};
exports.deleteAgentUploadFileHandler = deleteAgentUploadFileHandler;