"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getEnrichCommands = void 0;
exports.getMaxMinNumberOfParams = getMaxMinNumberOfParams;
exports.getSubqueriesToValidate = getSubqueriesToValidate;
var _esqlAst = require("@kbn/esql-ast");
var _expand_evals = require("../shared/expand_evals");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * Returns the maximum and minimum number of parameters allowed by a function
 *
 * Used for too-many, too-few arguments validation
 */
function getMaxMinNumberOfParams(definition) {
  if (definition.signatures.length === 0) {
    return {
      min: 0,
      max: 0
    };
  }
  let min = Infinity;
  let max = 0;
  definition.signatures.forEach(({
    params,
    minParams
  }) => {
    min = Math.min(min, params.filter(({
      optional
    }) => !optional).length);
    max = Math.max(max, minParams ? Infinity : params.length);
  });
  return {
    min,
    max
  };
}

/**
 * Collects all 'enrich' commands from a list of ESQL commands.
 * @param commands - The list of ESQL commands to search through.
 * This function traverses the provided ESQL commands and collects all commands with the name 'enrich'.
 * @returns {ESQLCommand[]} - An array of ESQLCommand objects that represent the 'enrich' commands found in the input.
 */
const getEnrichCommands = commands => _esqlAst.Walker.matchAll(commands, {
  type: 'command',
  name: 'enrich'
});

/**
 * Returns a list of subqueries to validate
 * @param rootCommands
 */
exports.getEnrichCommands = getEnrichCommands;
function getSubqueriesToValidate(rootCommands) {
  const subsequences = [];
  const expandedCommands = (0, _expand_evals.expandEvals)(rootCommands);
  for (let i = 0; i < expandedCommands.length; i++) {
    const command = expandedCommands[i];

    // every command within FORK's branches is its own subquery to be validated
    if (command.name.toLowerCase() === 'fork') {
      const branchSubqueries = getForkBranchSubqueries(command);
      for (const subquery of branchSubqueries) {
        subsequences.push([...expandedCommands.slice(0, i), ...subquery]);
      }
    }
    subsequences.push(expandedCommands.slice(0, i + 1));
  }
  return subsequences.map(subsequence => _esqlAst.Builder.expression.query(subsequence));
}

/**
 * Expands a FORK command into flat subqueries for each command in each branch.
 *
 * E.g. FORK (EVAL 1 | LIMIT 10) (RENAME foo AS bar | DROP lolz)
 *
 * becomes [`EVAL 1`, `EVAL 1 | LIMIT 10`, `RENAME foo AS bar`, `RENAME foo AS bar | DROP lolz`]
 *
 * @param command a FORK command
 * @returns an array of expanded subqueries
 */
function getForkBranchSubqueries(command) {
  const expanded = [];
  const branches = command.args;
  for (let j = 0; j < branches.length; j++) {
    for (let k = 0; k < branches[j].commands.length; k++) {
      const partialQuery = branches[j].commands.slice(0, k + 1);
      expanded.push(partialQuery);
    }
  }
  return expanded;
}