"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.indexInfoHandlerFactory = void 0;
var _i18n = require("@kbn/i18n");
var _stream_reducer = require("@kbn/aiops-log-rate-analysis/api/stream_reducer");
var _is_request_aborted_error = require("@kbn/aiops-common/is_request_aborted_error");
var _fetch_index_info = require("@kbn/aiops-log-rate-analysis/queries/fetch_index_info");
var _constants = require("../response_stream_utils/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const indexInfoHandlerFactory = options => async () => {
  const {
    abortSignal,
    esClient,
    logDebugMessage,
    logger,
    requestBody,
    responseStream,
    stateHandler,
    version
  } = options;
  const keywordFieldCandidates = [];
  let keywordFieldCandidatesCount = keywordFieldCandidates.length;
  const textFieldCandidates = [];
  let textFieldCandidatesCount = textFieldCandidates.length;
  let zeroDocsFallback = false;
  logDebugMessage('Fetch index information.');
  responseStream.push((0, _stream_reducer.updateLoadingState)({
    ccsWarning: false,
    loaded: stateHandler.loaded(),
    loadingState: _i18n.i18n.translate('xpack.aiops.logRateAnalysis.loadingState.loadingIndexInformation', {
      defaultMessage: 'Loading index information.'
    })
  }));
  let skipFieldCandidates = false;
  if (version === '2') {
    var _overrides;
    skipFieldCandidates = Array.isArray((_overrides = requestBody.overrides) === null || _overrides === void 0 ? void 0 : _overrides.remainingFieldCandidates);
  } else if (version === '3') {
    var _overrides2, _overrides3;
    skipFieldCandidates = Array.isArray((_overrides2 = requestBody.overrides) === null || _overrides2 === void 0 ? void 0 : _overrides2.remainingKeywordFieldCandidates) || Array.isArray((_overrides3 = requestBody.overrides) === null || _overrides3 === void 0 ? void 0 : _overrides3.remainingTextFieldCandidates);
  }
  try {
    const indexInfo = await (0, _fetch_index_info.fetchIndexInfo)({
      esClient,
      abortSignal,
      arguments: {
        ...requestBody,
        textFieldCandidatesOverrides: ['message', 'error.message'],
        skipFieldCandidates
      }
    });
    logDebugMessage(`Baseline document count: ${indexInfo.baselineTotalDocCount}`);
    logDebugMessage(`Deviation document count: ${indexInfo.deviationTotalDocCount}`);
    keywordFieldCandidates.push(...indexInfo.keywordFieldCandidates);
    keywordFieldCandidatesCount = keywordFieldCandidates.length;
    textFieldCandidates.push(...indexInfo.textFieldCandidates);
    textFieldCandidatesCount = textFieldCandidates.length;
    zeroDocsFallback = indexInfo.zeroDocsFallback;
  } catch (e) {
    if (!(0, _is_request_aborted_error.isRequestAbortedError)(e)) {
      logger.error(`Failed to fetch index information, got: \n${e.toString()}`);
      responseStream.pushError(`Failed to fetch index information.`);
    }
    responseStream.end();
    return;
  }
  stateHandler.loaded(_constants.LOADED_FIELD_CANDIDATES, false);
  responseStream.pushPingWithTimeout();
  responseStream.push((0, _stream_reducer.updateLoadingState)({
    ccsWarning: false,
    loaded: stateHandler.loaded(),
    loadingState: _i18n.i18n.translate('xpack.aiops.logRateAnalysis.loadingState.identifiedFieldCandidates', {
      defaultMessage: 'Identified {fieldCandidatesCount, plural, one {# field candidate} other {# field candidates}}.',
      values: {
        fieldCandidatesCount: keywordFieldCandidatesCount + textFieldCandidatesCount
      }
    })
  }));
  responseStream.push((0, _stream_reducer.setZeroDocsFallback)(zeroDocsFallback));
  if (!skipFieldCandidates && keywordFieldCandidatesCount === 0 && textFieldCandidatesCount === 0) {
    responseStream.endWithUpdatedLoadingState();
  } else if (stateHandler.shouldStop()) {
    logDebugMessage('shouldStop after fetching field candidates.');
    responseStream.end();
    return;
  }
  return {
    keywordFieldCandidates,
    textFieldCandidates,
    zeroDocsFallback
  };
};
exports.indexInfoHandlerFactory = indexInfoHandlerFactory;