"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getInstallCommandForPlatform = getInstallCommandForPlatform;
var _manual = require("../../../../../components/enrollment_instructions/manual");
var _hooks = require("../../../hooks");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function getArtifact(platform, kibanaVersion, downloadSource, downloadSourceProxy) {
  const ARTIFACT_BASE_URL = `${(0, _manual.getDownloadBaseUrl)(downloadSource)}/beats/elastic-agent`;
  const {
    windows: windowsDownloadSourceProxyArgs,
    curl: curlDownloadSourceProxyArgs
  } = (0, _manual.getDownloadSourceProxyArgs)(downloadSourceProxy);
  const appendWindowsDownloadSourceProxyArgs = windowsDownloadSourceProxyArgs ? ` ${windowsDownloadSourceProxyArgs}` : '';
  const appendCurlDownloadSourceProxyArgs = curlDownloadSourceProxyArgs ? ` ${curlDownloadSourceProxyArgs}` : '';
  const artifactMap = {
    linux_aarch64: {
      downloadCommand: [`curl -L -O ${ARTIFACT_BASE_URL}/elastic-agent-${kibanaVersion}-linux-arm64.tar.gz${appendCurlDownloadSourceProxyArgs}`, `tar xzvf elastic-agent-${kibanaVersion}-linux-arm64.tar.gz`, `cd elastic-agent-${kibanaVersion}-linux-arm64`].join(`\n`)
    },
    linux_x86_64: {
      downloadCommand: [`curl -L -O ${ARTIFACT_BASE_URL}/elastic-agent-${kibanaVersion}-linux-x86_64.tar.gz${appendCurlDownloadSourceProxyArgs}`, `tar xzvf elastic-agent-${kibanaVersion}-linux-x86_64.tar.gz`, `cd elastic-agent-${kibanaVersion}-linux-x86_64`].join(`\n`)
    },
    mac_aarch64: {
      downloadCommand: [`curl -L -O ${ARTIFACT_BASE_URL}/elastic-agent-${kibanaVersion}-darwin-aarch64.tar.gz${appendCurlDownloadSourceProxyArgs}`, `tar xzvf elastic-agent-${kibanaVersion}-darwin-aarch64.tar.gz`, `cd elastic-agent-${kibanaVersion}-darwin-aarch64`].join(`\n`)
    },
    mac_x86_64: {
      downloadCommand: [`curl -L -O ${ARTIFACT_BASE_URL}/elastic-agent-${kibanaVersion}-darwin-x86_64.tar.gz${appendCurlDownloadSourceProxyArgs}`, `tar xzvf elastic-agent-${kibanaVersion}-darwin-x86_64.tar.gz`, `cd elastic-agent-${kibanaVersion}-darwin-x86_64`].join(`\n`)
    },
    windows: {
      downloadCommand: [`$ProgressPreference = 'SilentlyContinue'`, `Invoke-WebRequest -Uri ${ARTIFACT_BASE_URL}/elastic-agent-${kibanaVersion}-windows-x86_64.zip -OutFile elastic-agent-${kibanaVersion}-windows-x86_64.zip${appendWindowsDownloadSourceProxyArgs}`, `Expand-Archive .\\elastic-agent-${kibanaVersion}-windows-x86_64.zip`, `cd elastic-agent-${kibanaVersion}-windows-x86_64`].join(`\n`)
    },
    windows_msi: {
      downloadCommand: [`$ProgressPreference = 'SilentlyContinue'`, `Invoke-WebRequest -Uri ${ARTIFACT_BASE_URL}/elastic-agent-${kibanaVersion}-windows-x86_64.msi -OutFile elastic-agent-${kibanaVersion}-windows-x86_64.msi${appendWindowsDownloadSourceProxyArgs}`].join(`\n`)
    },
    deb_aarch64: {
      downloadCommand: [`curl -L -O ${ARTIFACT_BASE_URL}/elastic-agent-${kibanaVersion}-arm64.deb${appendCurlDownloadSourceProxyArgs}`, `sudo ELASTIC_AGENT_FLAVOR=servers dpkg -i elastic-agent-${kibanaVersion}-arm64.deb`].join(`\n`)
    },
    deb_x86_64: {
      downloadCommand: [`curl -L -O ${ARTIFACT_BASE_URL}/elastic-agent-${kibanaVersion}-amd64.deb${appendCurlDownloadSourceProxyArgs}`, `sudo ELASTIC_AGENT_FLAVOR=servers dpkg -i elastic-agent-${kibanaVersion}-amd64.deb`].join(`\n`)
    },
    rpm_aarch64: {
      downloadCommand: [`curl -L -O ${ARTIFACT_BASE_URL}/elastic-agent-${kibanaVersion}-aarch64.rpm${appendCurlDownloadSourceProxyArgs}`, `sudo ELASTIC_AGENT_FLAVOR=servers rpm -vi elastic-agent-${kibanaVersion}-aarch64.rpm`].join(`\n`)
    },
    rpm_x86_64: {
      downloadCommand: [`curl -L -O ${ARTIFACT_BASE_URL}/elastic-agent-${kibanaVersion}-x86_64.rpm${appendCurlDownloadSourceProxyArgs}`, `sudo ELASTIC_AGENT_FLAVOR=servers rpm -vi elastic-agent-${kibanaVersion}-x86_64.rpm`].join(`\n`)
    },
    kubernetes: {
      downloadCommand: ''
    }
  };
  return artifactMap[platform];
}
function getInstallCommandForPlatform({
  platform,
  esOutputHost,
  serviceToken,
  esOutputProxy,
  policyId,
  fleetServerHost,
  isProductionDeployment,
  sslCATrustedFingerprint,
  kibanaVersion,
  downloadSource,
  downloadSourceProxy
}) {
  const newLineSeparator = platform === 'windows' || platform === 'windows_msi' ? '`\n' : '\\\n';
  const artifact = getArtifact(platform, kibanaVersion !== null && kibanaVersion !== void 0 ? kibanaVersion : '', downloadSource, downloadSourceProxy);
  const commandArguments = [];
  if (isProductionDeployment && fleetServerHost) {
    commandArguments.push(['url', fleetServerHost === null || fleetServerHost === void 0 ? void 0 : fleetServerHost.host_urls[0]]);
  }
  commandArguments.push(['fleet-server-es', esOutputHost]);
  commandArguments.push(['fleet-server-service-token', serviceToken]);
  if (policyId) {
    commandArguments.push(['fleet-server-policy', policyId]);
  }
  if (sslCATrustedFingerprint) {
    commandArguments.push(['fleet-server-es-ca-trusted-fingerprint', sslCATrustedFingerprint]);
  }
  if (isProductionDeployment) {
    var _fleetServerHost$ssl, _fleetServerHost$ssl2, _fleetServerHost$ssl3, _fleetServerHost$ssl4;
    const certificateAuthorities = fleetServerHost !== null && fleetServerHost !== void 0 && (_fleetServerHost$ssl = fleetServerHost.ssl) !== null && _fleetServerHost$ssl !== void 0 && _fleetServerHost$ssl.certificate_authorities ? `'${fleetServerHost === null || fleetServerHost === void 0 ? void 0 : (_fleetServerHost$ssl2 = fleetServerHost.ssl) === null || _fleetServerHost$ssl2 === void 0 ? void 0 : _fleetServerHost$ssl2.certificate_authorities}'` : '<PATH_TO_CA>';
    const fleetServerCert = fleetServerHost !== null && fleetServerHost !== void 0 && (_fleetServerHost$ssl3 = fleetServerHost.ssl) !== null && _fleetServerHost$ssl3 !== void 0 && _fleetServerHost$ssl3.certificate ? `'${fleetServerHost === null || fleetServerHost === void 0 ? void 0 : (_fleetServerHost$ssl4 = fleetServerHost.ssl) === null || _fleetServerHost$ssl4 === void 0 ? void 0 : _fleetServerHost$ssl4.certificate}'` : '<PATH_TO_FLEET_SERVER_CERT>';
    commandArguments.push(['certificate-authorities', certificateAuthorities]);
    if (!sslCATrustedFingerprint) {
      var _fleetServerHost$ssl5, _fleetServerHost$ssl6;
      const esCert = fleetServerHost !== null && fleetServerHost !== void 0 && (_fleetServerHost$ssl5 = fleetServerHost.ssl) !== null && _fleetServerHost$ssl5 !== void 0 && _fleetServerHost$ssl5.es_certificate ? `'${fleetServerHost === null || fleetServerHost === void 0 ? void 0 : (_fleetServerHost$ssl6 = fleetServerHost.ssl) === null || _fleetServerHost$ssl6 === void 0 ? void 0 : _fleetServerHost$ssl6.es_certificate}'` : '<PATH_TO_ES_CERT>';
      commandArguments.push(['fleet-server-es-ca', esCert]);
    }
    commandArguments.push(['fleet-server-cert', fleetServerCert]);
    commandArguments.push(['fleet-server-cert-key', '<PATH_TO_FLEET_SERVER_CERT_KEY>']);
  }
  commandArguments.push(['fleet-server-port', '8220']);
  if (_hooks.PLATFORM_WITH_INSTALL_SERVERS.includes(platform)) {
    commandArguments.push(['install-servers']);
  }
  const enrollmentProxyArgs = [];
  if (esOutputProxy) {
    enrollmentProxyArgs.push(['proxy-url', esOutputProxy.url]);
    Object.entries(esOutputProxy.proxy_headers || []).forEach(([key, value]) => {
      enrollmentProxyArgs.push(['proxy-header', `"${key}=${value}"`]);
    });
  }
  const commandArgumentsStr = [...commandArguments, ...enrollmentProxyArgs].reduce((acc, [key, val]) => {
    if (acc === '' && key === 'url') {
      return `--${key}=${val}`;
    }
    const valOrEmpty = val ? `=${val}` : '';
    return acc += ` ${newLineSeparator}  --${key}${valOrEmpty}`;
  }, '').trim();
  const commands = {
    linux_aarch64: `${artifact.downloadCommand}\nsudo ./elastic-agent install ${commandArgumentsStr}`,
    linux_x86_64: `${artifact.downloadCommand}\nsudo ./elastic-agent install ${commandArgumentsStr}`,
    mac_aarch64: `${artifact.downloadCommand}\nsudo ./elastic-agent install ${commandArgumentsStr}`,
    mac_x86_64: `${artifact.downloadCommand}\nsudo ./elastic-agent install ${commandArgumentsStr}`,
    windows: `${artifact.downloadCommand}\n.\\elastic-agent.exe install ${commandArgumentsStr}`,
    windows_msi: `${artifact.downloadCommand}\n.\\elastic-agent-${kibanaVersion}-windows-x86_64.msi --% INSTALLARGS="${commandArgumentsStr}"`,
    deb_aarch64: `${artifact.downloadCommand}\nsudo systemctl enable elastic-agent\nsudo systemctl start elastic-agent\nsudo elastic-agent enroll ${commandArgumentsStr}`,
    deb_x86_64: `${artifact.downloadCommand}\nsudo systemctl enable elastic-agent\nsudo systemctl start elastic-agent\nsudo elastic-agent enroll ${commandArgumentsStr}`,
    rpm_aarch64: `${artifact.downloadCommand}\nsudo systemctl enable elastic-agent\nsudo systemctl start elastic-agent\nsudo elastic-agent enroll ${commandArgumentsStr}`,
    rpm_x86_64: `${artifact.downloadCommand}\nsudo systemctl enable elastic-agent\nsudo systemctl start elastic-agent\nsudo elastic-agent enroll ${commandArgumentsStr}`,
    kubernetes: '',
    cloudFormation: '',
    googleCloudShell: ''
  };
  return commands[platform];
}