"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.resolveProviderAndModel = exports.getPromptsByGroupId = exports.getPrompt = void 0;
var _inferenceCommon = require("@kbn/inference-common");
var _utils = require("./utils");
var _saved_object_mappings = require("./saved_object_mappings");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Get prompts by feature (promptGroupId)
 * provide either model + provider or connector to avoid additional calls to get connector
 * @param actionsClient - actions client (look up connector if connector is not provided)
 * @param connector - connector, provide if available. No need to provide model and provider in this case
 * @param connectorId - connector id
 * @param localPrompts - local prompts object
 * @param model - model. No need to provide if connector provided
 * @param promptGroupId - feature id, should be common across promptIds
 * @param promptIds - prompt ids with shared promptGroupId
 * @param provider  - provider. No need to provide if connector provided
 * @param savedObjectsClient - saved objects client
 */
const getPromptsByGroupId = async ({
  actionsClient,
  connector,
  connectorId,
  localPrompts,
  model: providedModel,
  promptGroupId,
  promptIds,
  provider: providedProvider,
  savedObjectsClient
}) => {
  var _prompts$saved_object;
  const {
    provider,
    model
  } = await resolveProviderAndModel({
    providedProvider,
    providedModel,
    connectorId,
    actionsClient,
    providedConnector: connector
  });
  const prompts = await savedObjectsClient.find({
    type: _saved_object_mappings.promptSavedObjectType,
    searchFields: ['promptGroupId'],
    search: promptGroupId
  });
  const promptsOnly = (_prompts$saved_object = prompts === null || prompts === void 0 ? void 0 : prompts.saved_objects.map(p => p.attributes)) !== null && _prompts$saved_object !== void 0 ? _prompts$saved_object : [];
  return promptIds.map(promptId => {
    var _promptsOnly$filter;
    const prompt = findPromptEntry({
      prompts: (_promptsOnly$filter = promptsOnly.filter(p => p.promptId === promptId)) !== null && _promptsOnly$filter !== void 0 ? _promptsOnly$filter : [],
      promptId,
      promptGroupId,
      provider,
      model,
      localPrompts
    });
    if (!prompt) {
      throw new Error(`Prompt not found for promptId: ${promptId} and promptGroupId: ${promptGroupId}`);
    }
    return {
      promptId,
      prompt
    };
  });
};

/**
 * Get prompt by promptId
 * provide either model + provider or connector to avoid additional calls to get connector
 * @param actionsClient - actions client
 * @param connector - connector, provide if available. No need to provide model and provider in this case
 * @param connectorId - connector id
 * @param localPrompts - local prompts object
 * @param model - model. No need to provide if connector provided
 * @param promptId - prompt id
 * @param promptGroupId - feature id, should be common across promptIds
 * @param provider  - provider. No need to provide if connector provided
 * @param savedObjectsClient - saved objects client
 */
exports.getPromptsByGroupId = getPromptsByGroupId;
const getPrompt = async ({
  actionsClient,
  connector,
  connectorId,
  localPrompts,
  model: providedModel,
  promptGroupId,
  promptId,
  provider: providedProvider,
  savedObjectsClient
}) => {
  var _prompts$saved_object2;
  const {
    provider,
    model
  } = await resolveProviderAndModel({
    providedProvider,
    providedModel,
    connectorId,
    actionsClient,
    providedConnector: connector
  });
  const prompts = await savedObjectsClient.find({
    type: _saved_object_mappings.promptSavedObjectType,
    filter: `${_saved_object_mappings.promptSavedObjectType}.attributes.promptId: "${promptId}" AND ${_saved_object_mappings.promptSavedObjectType}.attributes.promptGroupId: "${promptGroupId}"`,
    fields: ['provider', 'model', 'prompt']
  });
  const prompt = findPromptEntry({
    prompts: (_prompts$saved_object2 = prompts === null || prompts === void 0 ? void 0 : prompts.saved_objects.map(p => p.attributes)) !== null && _prompts$saved_object2 !== void 0 ? _prompts$saved_object2 : [],
    promptId,
    promptGroupId,
    provider,
    model,
    localPrompts
  });
  if (!prompt) {
    throw new Error(`Prompt not found for promptId: ${promptId} and promptGroupId: ${promptGroupId}`);
  }
  return prompt;
};
exports.getPrompt = getPrompt;
const resolveProviderAndModel = async ({
  providedProvider,
  providedModel,
  connectorId,
  actionsClient,
  providedConnector
}) => {
  let model = providedModel;
  let provider = providedProvider;
  if (!provider || !model || provider === 'inference') {
    let connector = providedConnector;
    if (!connector && connectorId != null && actionsClient) {
      connector = await actionsClient.get({
        id: connectorId
      });
    }
    if (!connector) {
      return {};
    }
    if (provider === 'inference' && connector.config) {
      var _connector$config$pro;
      provider = connector.config.provider || provider;
      model = ((_connector$config$pro = connector.config.providerConfig) === null || _connector$config$pro === void 0 ? void 0 : _connector$config$pro.model_id) || model;
      if (provider === 'elastic' && model) {
        var _elasticModelDictiona, _elasticModelDictiona2;
        provider = ((_elasticModelDictiona = _inferenceCommon.elasticModelDictionary[model]) === null || _elasticModelDictiona === void 0 ? void 0 : _elasticModelDictiona.provider) || 'inference';
        model = (_elasticModelDictiona2 = _inferenceCommon.elasticModelDictionary[model]) === null || _elasticModelDictiona2 === void 0 ? void 0 : _elasticModelDictiona2.model;
      }
    } else if (connector.config) {
      provider = provider || (0, _utils.getProviderFromActionTypeId)(connector.actionTypeId);
      model = model || connector.config.defaultModel;
    }
  }
  return {
    provider: provider === 'inference' ? 'bedrock' : provider,
    model
  };
};
exports.resolveProviderAndModel = resolveProviderAndModel;
const findPrompt = ({
  prompts,
  conditions
}) => {
  for (const condition of conditions) {
    const match = prompts.find(condition);
    if (match) return match.prompt.default;
  }
  return undefined;
};
const findPromptEntry = ({
  prompts,
  promptId,
  promptGroupId,
  provider,
  model,
  localPrompts
}) => {
  var _findPrompt;
  const conditions = [prompt => prompt.provider === provider && prompt.model === model, prompt => prompt.provider === provider && !prompt.model, prompt => !prompt.provider && !prompt.model];
  return (_findPrompt = findPrompt({
    prompts,
    conditions
  })) !== null && _findPrompt !== void 0 ? _findPrompt : findPrompt({
    prompts: localPrompts.filter(p => p.promptId === promptId && p.promptGroupId === promptGroupId),
    conditions
  });
};