"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getDataStreams = getDataStreams;
var _elasticsearch = require("@elastic/elasticsearch");
var _data_streams = require("../../data_streams");
var _errors = require("../../../errors");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const VALID_STREAM_TYPES = ['logs', 'metrics', 'traces', 'synthetics', 'profiling'];
async function getDataStreams(options) {
  const {
    esClient,
    type,
    datasetQuery,
    uncategorisedOnly,
    sortOrder
  } = options;
  const allDataStreams = await _data_streams.dataStreamService.getMatchingDataStreams(esClient, {
    type: type ? type : '*',
    dataset: datasetQuery ? `*${datasetQuery}*` : '*'
  }).catch(err => {
    var _err$body, _err$body$error;
    const isResponseError = err instanceof _elasticsearch.errors.ResponseError;
    if (isResponseError && (err === null || err === void 0 ? void 0 : (_err$body = err.body) === null || _err$body === void 0 ? void 0 : (_err$body$error = _err$body.error) === null || _err$body$error === void 0 ? void 0 : _err$body$error.type) === 'security_exception') {
      throw new _errors.FleetUnauthorizedError(`Unauthorized to query datastreams: ${err.message}`);
    }
    throw err;
  });
  let filteredDataStreams = uncategorisedOnly ? allDataStreams.filter(stream => {
    return !stream._meta || !stream._meta.managed_by || stream._meta.managed_by !== 'fleet';
  }) : allDataStreams;
  filteredDataStreams = filteredDataStreams.filter(stream => {
    const isValidStreamType = VALID_STREAM_TYPES.some(streamType => stream.name.startsWith(streamType));
    return isValidStreamType;
  });
  const mappedDataStreams = filteredDataStreams.map(dataStream => {
    return {
      name: dataStream.name
    };
  });
  const sortedDataStreams = mappedDataStreams.sort((a, b) => {
    return a.name.localeCompare(b.name);
  });
  const dataStreams = sortOrder === 'asc' ? sortedDataStreams : sortedDataStreams.reverse();
  return {
    items: dataStreams
  };
}