"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.updateCloudConnectorHandler = exports.getCloudConnectorsHandler = exports.getCloudConnectorHandler = exports.deleteCloudConnectorHandler = exports.createCloudConnectorHandler = void 0;
var _services = require("../../services");
var _app_context = require("../../services/app_context");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const createCloudConnectorHandler = async (context, request, response) => {
  const fleetContext = await context.fleet;
  const {
    internalSoClient
  } = fleetContext;
  const logger = _app_context.appContextService.getLogger().get('CloudConnectorService createCloudConnectorHandler');
  try {
    logger.info('Creating cloud connector');
    const cloudConnector = await _services.cloudConnectorService.create(internalSoClient, request.body);
    logger.info(`Successfully created cloud connector ${cloudConnector.id}`);
    const body = {
      item: cloudConnector
    };
    return response.ok({
      body
    });
  } catch (error) {
    logger.error(`Failed to create cloud connector`, error.message);
    return response.customError({
      statusCode: 400,
      body: {
        message: error.message
      }
    });
  }
};
exports.createCloudConnectorHandler = createCloudConnectorHandler;
const getCloudConnectorsHandler = async (context, request, response) => {
  const fleetContext = await context.fleet;
  const {
    internalSoClient
  } = fleetContext;
  const {
    page,
    perPage
  } = request.query;
  const logger = _app_context.appContextService.getLogger().get('CloudConnectorService getCloudConnectorsHandler');
  try {
    logger.info('Getting cloud connectors list');
    const cloudConnectors = await _services.cloudConnectorService.getList(internalSoClient, {
      page: page ? parseInt(page, 10) : undefined,
      perPage: perPage ? parseInt(perPage, 10) : undefined
    });
    logger.info('Successfully retrieved cloud connectors list');
    const body = {
      items: cloudConnectors
    };
    return response.ok({
      body
    });
  } catch (error) {
    logger.error('Failed to get cloud connectors list', error.message);
    return response.customError({
      statusCode: 400,
      body: {
        message: error.message
      }
    });
  }
};
exports.getCloudConnectorsHandler = getCloudConnectorsHandler;
const getCloudConnectorHandler = async (context, request, response) => {
  const fleetContext = await context.fleet;
  const {
    internalSoClient
  } = fleetContext;
  const cloudConnectorId = request.params.cloudConnectorId;
  const logger = _app_context.appContextService.getLogger().get('CloudConnectorService getCloudConnectorHandler');
  try {
    logger.info(`Getting cloud connector ${cloudConnectorId}`);
    const result = await _services.cloudConnectorService.getById(internalSoClient, cloudConnectorId);
    logger.info(`Successfully retrieved cloud connector ${cloudConnectorId}`);
    const body = {
      item: result
    };
    return response.ok({
      body
    });
  } catch (error) {
    logger.error(`Failed to get cloud connector ${cloudConnectorId}`, error.message);
    return response.customError({
      statusCode: 400,
      body: {
        message: error.message
      }
    });
  }
};
exports.getCloudConnectorHandler = getCloudConnectorHandler;
const updateCloudConnectorHandler = async (context, request, response) => {
  const fleetContext = await context.fleet;
  const {
    internalSoClient
  } = fleetContext;
  const cloudConnectorId = request.params.cloudConnectorId;
  const logger = _app_context.appContextService.getLogger().get('CloudConnectorService updateCloudConnectorHandler');
  try {
    logger.info(`Updating cloud connector ${cloudConnectorId}`);
    const result = await _services.cloudConnectorService.update(internalSoClient, cloudConnectorId, request.body);
    logger.info(`Successfully updated cloud connector ${cloudConnectorId}`);
    const body = {
      item: result
    };
    return response.ok({
      body
    });
  } catch (error) {
    logger.error(`Failed to update cloud connector ${cloudConnectorId}`, error.message);
    return response.customError({
      statusCode: 400,
      body: {
        message: error.message
      }
    });
  }
};
exports.updateCloudConnectorHandler = updateCloudConnectorHandler;
const deleteCloudConnectorHandler = async (context, request, response) => {
  const fleetContext = await context.fleet;
  const {
    internalSoClient
  } = fleetContext;
  const cloudConnectorId = request.params.cloudConnectorId;
  const force = request.query.force || false;
  const logger = _app_context.appContextService.getLogger().get('CloudConnectorService deleteCloudConnectorHandler');
  try {
    logger.info(`Deleting cloud connector ${cloudConnectorId} (force: ${force})`);
    const result = await _services.cloudConnectorService.delete(internalSoClient, cloudConnectorId, force);
    logger.info(`Successfully deleted cloud connector ${cloudConnectorId}`);
    const body = {
      id: result.id
    };
    return response.ok({
      body
    });
  } catch (error) {
    logger.error(`Failed to delete cloud connector ${cloudConnectorId}`, error.message);
    return response.customError({
      statusCode: 400,
      body: {
        message: error.message
      }
    });
  }
};
exports.deleteCloudConnectorHandler = deleteCloudConnectorHandler;