"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createAttackDiscoverySchedulesInternalRoute = void 0;
var _securitysolutionEsUtils = require("@kbn/securitysolution-es-utils");
var _actions = require("@kbn/security-solution-features/actions");
var _zodHelpers = require("@kbn/zod-helpers");
var _elasticAssistantCommon = require("@kbn/elastic-assistant-common");
var _build_response = require("../../../../../lib/build_response");
var _helpers = require("../../../../helpers");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const createAttackDiscoverySchedulesInternalRoute = router => {
  router.versioned.post({
    access: 'internal',
    path: _elasticAssistantCommon.ATTACK_DISCOVERY_INTERNAL_SCHEDULES,
    security: {
      authz: {
        requiredPrivileges: ['elasticAssistant', _actions.ATTACK_DISCOVERY_API_ACTION_UPDATE_ATTACK_DISCOVERY_SCHEDULE]
      }
    }
  }).addVersion({
    version: _elasticAssistantCommon.API_VERSIONS.internal.v1,
    validate: {
      request: {
        body: (0, _zodHelpers.buildRouteValidationWithZod)(_elasticAssistantCommon.CreateAttackDiscoverySchedulesInternalRequestBody)
      },
      response: {
        200: {
          body: {
            custom: (0, _zodHelpers.buildRouteValidationWithZod)(_elasticAssistantCommon.CreateAttackDiscoverySchedulesInternalResponse)
          }
        }
      }
    }
  }, async (context, request, response) => {
    const ctx = await context.resolve(['core', 'elasticAssistant', 'licensing']);
    const resp = (0, _build_response.buildResponse)(response);
    const assistantContext = await context.elasticAssistant;
    const logger = assistantContext.logger;

    // Perform license and authenticated user
    const checkResponse = await (0, _helpers.performChecks)({
      context: ctx,
      request,
      response
    });
    if (!checkResponse.isSuccess) {
      return checkResponse.response;
    }
    try {
      const dataClient = await assistantContext.getAttackDiscoverySchedulingDataClient();
      if (!dataClient) {
        return resp.error({
          body: `Attack discovery data client not initialized`,
          statusCode: 500
        });
      }
      const schedule = await dataClient.createSchedule(request.body);
      return response.ok({
        body: schedule
      });
    } catch (err) {
      logger.error(err);
      const error = (0, _securitysolutionEsUtils.transformError)(err);
      return resp.error({
        body: {
          success: false,
          error: error.message
        },
        statusCode: error.statusCode
      });
    }
  });
};
exports.createAttackDiscoverySchedulesInternalRoute = createAttackDiscoverySchedulesInternalRoute;