"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.validRouteSecurity = void 0;
var _configSchema = require("@kbn/config-schema");
var _coreHttpServer = require("@kbn/core-http-server");
var _coreSecurityServer = require("@kbn/core-security-server");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const privilegeSetSchema = _configSchema.schema.object({
  anyRequired: _configSchema.schema.maybe(_configSchema.schema.arrayOf(_configSchema.schema.oneOf([_configSchema.schema.string(), _configSchema.schema.object({
    allOf: _configSchema.schema.arrayOf(_configSchema.schema.string(), {
      minSize: 2
    })
  })]), {
    minSize: 2
  })),
  allRequired: _configSchema.schema.maybe(_configSchema.schema.arrayOf(_configSchema.schema.oneOf([_configSchema.schema.string(), _configSchema.schema.object({
    anyOf: _configSchema.schema.arrayOf(_configSchema.schema.string(), {
      minSize: 2
    })
  })]), {
    minSize: 1
  }))
}, {
  validate: value => {
    if (!value.anyRequired && !value.allRequired) {
      return 'either anyRequired or allRequired must be specified';
    }
  }
});
const requiredPrivilegesSchema = _configSchema.schema.arrayOf(_configSchema.schema.oneOf([privilegeSetSchema, _configSchema.schema.string()]), {
  validate: value => {
    const anyRequired = [];
    const allRequired = [];
    if (!Array.isArray(value)) {
      return undefined;
    }
    value.forEach(privilege => {
      if (typeof privilege === 'string') {
        allRequired.push(privilege);
      } else {
        if (privilege.anyRequired) {
          anyRequired.push(...(0, _coreSecurityServer.unwindNestedSecurityPrivileges)(privilege.anyRequired));
        }
        if (privilege.allRequired) {
          allRequired.push(...(0, _coreSecurityServer.unwindNestedSecurityPrivileges)(privilege.allRequired));
        }
      }
    });
    if (anyRequired.includes(_coreHttpServer.ReservedPrivilegesSet.superuser)) {
      return 'Using superuser privileges in anyRequired is not allowed';
    }
    const hasSuperuserInAllRequired = allRequired.includes(_coreHttpServer.ReservedPrivilegesSet.superuser);
    const hasOperatorInAllRequired = allRequired.includes(_coreHttpServer.ReservedPrivilegesSet.operator);

    // Combining superuser with other privileges is redundant.
    // If user is a superuser, they inherently have access to all the privileges that may come with other roles.
    // The exception is when superuser and operator are the only required privileges.
    if (hasSuperuserInAllRequired && allRequired.length > 1 && !(hasOperatorInAllRequired && allRequired.length === 2)) {
      return 'Combining superuser with other privileges is redundant, superuser privileges set can be only used as a standalone privilege.';
    }

    // Operator privilege requires at least one additional non-operator privilege to be defined, that's why it's not allowed in anyRequired.
    if (anyRequired.includes(_coreHttpServer.ReservedPrivilegesSet.operator)) {
      return 'Using operator privileges in anyRequired is not allowed';
    }
    if (hasOperatorInAllRequired && allRequired.length === 1) {
      return 'Operator privilege requires at least one additional non-operator privilege to be defined';
    }
    if (anyRequired.length && allRequired.length) {
      for (const privilege of anyRequired) {
        if (allRequired.includes(privilege)) {
          return `anyRequired and allRequired cannot have the same values: [${privilege}]`;
        }
      }
    }
    if (anyRequired.length) {
      const uniqueAnyPrivileges = new Set([...anyRequired]);
      if (anyRequired.length !== uniqueAnyPrivileges.size) {
        return 'anyRequired privileges must contain unique values';
      }
    }
    if (allRequired.length) {
      const uniqueAllPrivileges = new Set([...allRequired]);
      if (allRequired.length !== uniqueAllPrivileges.size) {
        return 'allRequired privileges must contain unique values';
      }
    }
  },
  minSize: 1
});
const authzSchema = _configSchema.schema.object({
  enabled: _configSchema.schema.maybe(_configSchema.schema.literal(false)),
  requiredPrivileges: _configSchema.schema.conditional(_configSchema.schema.siblingRef('enabled'), _configSchema.schema.never(), requiredPrivilegesSchema, _configSchema.schema.never()),
  reason: _configSchema.schema.conditional(_configSchema.schema.siblingRef('enabled'), _configSchema.schema.never(), _configSchema.schema.never(), _configSchema.schema.string())
});
const authcSchema = _configSchema.schema.object({
  enabled: _configSchema.schema.oneOf([_configSchema.schema.literal(true), _configSchema.schema.literal('optional'), _configSchema.schema.literal(false)]),
  reason: _configSchema.schema.conditional(_configSchema.schema.siblingRef('enabled'), _configSchema.schema.literal(false), _configSchema.schema.string(), _configSchema.schema.never())
});
const routeSecuritySchema = _configSchema.schema.object({
  authz: authzSchema,
  authc: _configSchema.schema.maybe(authcSchema)
});
const validRouteSecurity = (routeSecurity, options) => {
  if (!routeSecurity) {
    return routeSecurity;
  }
  if ((routeSecurity === null || routeSecurity === void 0 ? void 0 : routeSecurity.authc) !== undefined && (options === null || options === void 0 ? void 0 : options.authRequired) !== undefined) {
    throw new Error('Cannot specify both security.authc and options.authRequired');
  }
  return routeSecuritySchema.validate(routeSecurity);
};
exports.validRouteSecurity = validRouteSecurity;