"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.CloudExperimentsPlugin = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _rxjs = require("rxjs");
var _serverSdk = require("@openfeature/server-sdk");
var _openfeatureNodeServer = require("@launchdarkly/openfeature-node-server");
var _metadata_service = require("../common/metadata_service");
var _usage = require("./usage");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class CloudExperimentsPlugin {
  constructor(initializerContext) {
    (0, _defineProperty2.default)(this, "logger", void 0);
    (0, _defineProperty2.default)(this, "metadataService", void 0);
    this.initializerContext = initializerContext;
    this.logger = initializerContext.logger.get();
    const config = initializerContext.config.get();
    this.metadataService = new _metadata_service.MetadataService({
      metadata_refresh_interval: config.metadata_refresh_interval
    }, this.logger.get('metadata'));
    const ldConfig = config.launch_darkly; // If the plugin is enabled and no flag_overrides are provided (dev mode only), launch_darkly must exist
    if (!ldConfig && !initializerContext.env.mode.dev) {
      // If the plugin is enabled, and it's in prod mode, launch_darkly must exist
      // (config-schema should enforce it, but just in case).
      throw new Error('xpack.cloud_integrations.experiments.launch_darkly configuration should exist');
    }
  }
  setup(core, deps) {
    // Ideally we should have something like this for the browser as well.
    core.logging.configure(this.initializerContext.config.create().pipe((0, _rxjs.map)(({
      launch_darkly: {
        client_log_level: clientLogLevel = 'none'
      } = {}
    }) => {
      const logLevel = clientLogLevel.replace('none', 'off');
      return {
        loggers: [{
          name: 'launch-darkly',
          level: logLevel,
          appenders: []
        }]
      };
    })));
    (0, _metadata_service.initializeMetadata)({
      metadataService: this.metadataService,
      initializerContext: this.initializerContext,
      cloud: deps.cloud,
      featureFlags: core.featureFlags,
      logger: this.logger
    });
    const launchDarklyOpenFeatureProvider = this.createOpenFeatureProvider();
    if (launchDarklyOpenFeatureProvider) {
      core.featureFlags.setProvider(launchDarklyOpenFeatureProvider);
      core.featureFlags.setInitialFeatureFlagsGetter(async () => {
        const launchDarklyClient = launchDarklyOpenFeatureProvider.getClient();
        const context = _serverSdk.OpenFeature.getContext();
        const {
          flags
        } = await (0, _usage.getAllFlags)(launchDarklyClient, context);
        return flags;
      });
    }
    (0, _usage.registerUsageCollector)(deps.usageCollection, () => ({
      launchDarklyClient: launchDarklyOpenFeatureProvider === null || launchDarklyOpenFeatureProvider === void 0 ? void 0 : launchDarklyOpenFeatureProvider.getClient(),
      currentContext: _serverSdk.OpenFeature.getContext()
    }));
  }
  start(core, deps) {
    this.metadataService.start({
      hasDataFetcher: async () => await this.addHasDataMetadata(core, deps.dataViews)
    });
  }
  stop() {
    this.metadataService.stop();
  }
  createOpenFeatureProvider() {
    const {
      launch_darkly: ldConfig
    } = this.initializerContext.config.get();
    if (!ldConfig) return;
    return new _openfeatureNodeServer.LaunchDarklyProvider(ldConfig.sdk_key, {
      logger: this.logger.get('launch-darkly'),
      application: {
        id: 'kibana-server',
        version: this.initializerContext.env.packageInfo.buildFlavor === 'serverless' ? this.initializerContext.env.packageInfo.buildSha : this.initializerContext.env.packageInfo.version
      }
    });
  }
  async addHasDataMetadata(core, dataViews) {
    const dataViewsService = await dataViews.dataViewsServiceFactory(core.savedObjects.createInternalRepository(), core.elasticsearch.client.asInternalUser, void 0,
    // No Kibana Request to scope the check
    true // Ignore capabilities checks
    );
    return {
      has_data: await dataViewsService.hasUserDataView()
    };
  }
}
exports.CloudExperimentsPlugin = CloudExperimentsPlugin;