"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getAllSpacesWithCases = getAllSpacesWithCases;
exports.isRetryableEsClientError = void 0;
var _elasticsearch = require("@elastic/elasticsearch");
var _constants = require("../../common/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const retryResponseStatuses = [401,
// AuthorizationException
403,
// AuthenticationException
408,
// RequestTimeout
410,
// Gone
429,
// TooManyRequests -> ES circuit breaker
503,
// ServiceUnavailable
504 // GatewayTimeout
];

/**
 * Returns true if the given elasticsearch error should be retried
 * by retry-based resiliency systems such as the SO migration, false otherwise.
 */
const isRetryableEsClientError = e => {
  var _e$body, _e$body$error;
  if (e instanceof _elasticsearch.errors.NoLivingConnectionsError || e instanceof _elasticsearch.errors.ConnectionError || e instanceof _elasticsearch.errors.TimeoutError || e instanceof _elasticsearch.errors.ResponseError && (e !== null && e !== void 0 && e.statusCode && retryResponseStatuses.includes(e === null || e === void 0 ? void 0 : e.statusCode) ||
  // ES returns a 400 Bad Request when trying to close or delete an
  // index while snapshots are in progress. This should have been a 503
  // so once https://github.com/elastic/elasticsearch/issues/65883 is
  // fixed we can remove this.
  (e === null || e === void 0 ? void 0 : (_e$body = e.body) === null || _e$body === void 0 ? void 0 : (_e$body$error = _e$body.error) === null || _e$body$error === void 0 ? void 0 : _e$body$error.type) === 'snapshot_in_progress_exception')) {
    return true;
  }
  return false;
};
exports.isRetryableEsClientError = isRetryableEsClientError;
const MAX_BUCKETS_LIMIT = 65535;
async function getAllSpacesWithCases(savedObjectsClient) {
  var _spaces$aggregations$, _spaces$aggregations;
  // This is one way to get all spaces that we want for case analytics purposes.
  // The advantage of this approach is that only spaces that actually contain cases are selected .
  // In turn that means that no unnecessary indices are created.
  // The disadvantage is that the query is an aggregation across the entire cluster and could be
  // slow in case there are many shards and a lot of cases.
  // The alternative is to query the list of all spaces in the cluster and thus creating extra
  // indices that might not be necessary.
  const spaces = await savedObjectsClient.find({
    type: _constants.CASE_SAVED_OBJECT,
    page: 0,
    perPage: 0,
    namespaces: ['*'],
    aggs: {
      spaces: {
        terms: {
          // We want to make sure that we include all spaces, because `terms` aggregations
          // by default only return the top 10 results. `MAX_BUCKETS_LIMIT` is 65k.
          size: MAX_BUCKETS_LIMIT,
          field: `${_constants.CASE_SAVED_OBJECT}.namespaces`
        }
      }
    }
  });
  return (_spaces$aggregations$ = (_spaces$aggregations = spaces.aggregations) === null || _spaces$aggregations === void 0 ? void 0 : _spaces$aggregations.spaces.buckets.map(space => space.key)) !== null && _spaces$aggregations$ !== void 0 ? _spaces$aggregations$ : [];
}