"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.defineRoutes = defineRoutes;
var _configSchema = require("@kbn/config-schema");
var _constants = require("../../common/constants");
var _notebook_catalog = require("../lib/notebook_catalog");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function defineRoutes({
  config,
  notebooksCache,
  logger,
  router
}) {
  router.get({
    path: '/internal/search_notebooks/notebooks',
    security: {
      authz: {
        enabled: false,
        reason: 'This route delegates authorization to the notebook service'
      }
    },
    validate: {
      query: _configSchema.schema.object({
        list: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    },
    options: {
      access: 'internal'
    }
  }, async (_context, request, response) => {
    const {
      list
    } = request.query;
    const resp = await (0, _notebook_catalog.getNotebookCatalog)({
      cache: notebooksCache,
      config,
      logger,
      notebookList: list
    });
    return response.ok({
      body: resp,
      headers: {
        'content-type': 'application/json'
      }
    });
  });
  router.get({
    path: '/internal/search_notebooks/notebooks/{notebookId}',
    security: {
      authz: {
        enabled: false,
        reason: 'This route delegates authorization to the notebook service'
      }
    },
    validate: {
      params: _configSchema.schema.object({
        notebookId: _configSchema.schema.string()
      })
    },
    options: {
      access: 'internal'
    }
  }, async (_, request, response) => {
    const notebookId = request.params.notebookId;
    if (notebookId === _constants.INTRODUCTION_NOTEBOOK.id) {
      return response.ok({
        body: _constants.INTRODUCTION_NOTEBOOK,
        headers: {
          'content-type': 'application/json'
        }
      });
    }
    const notebookMetadata = (0, _notebook_catalog.getNotebookMetadata)(notebookId, notebooksCache);
    if (!notebookMetadata) {
      logger.warn(`Unknown search notebook requested ${notebookId}`);
      return response.notFound();
    }
    let notebook;
    try {
      notebook = await (0, _notebook_catalog.getNotebook)(notebookId, {
        cache: notebooksCache,
        config,
        logger
      });
    } catch (e) {
      logger.warn(`Error getting search notebook ${notebookId}.`);
      logger.warn(e);
      return response.customError(e.message);
    }
    if (!notebook) {
      logger.warn(`Search notebook requested ${notebookId} not found or failed to fetch.`);
      return response.notFound();
    }
    return response.ok({
      body: {
        ...notebookMetadata,
        notebook
      },
      headers: {
        'content-type': 'application/json'
      }
    });
  });
}