"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getAllHosts = void 0;
var _server = require("@kbn/observability-plugin/server");
var _server2 = require("@kbn/metrics-data-access-plugin/server");
var _common = require("@kbn/metrics-data-access-plugin/common");
var _constants = require("../../../../../common/constants");
var _query = require("../helpers/query");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getAllHosts = async ({
  infraMetricsClient,
  apmDocumentSources,
  from,
  to,
  limit,
  metrics,
  hostNames,
  apmDataAccessServices,
  schema = 'ecs'
}) => {
  var _inventoryModel$nodeF, _inventoryModel$nodeF2, _response$aggregation, _response$aggregation2, _response$aggregation3, _response$aggregation4;
  const inventoryModel = (0, _common.findInventoryModel)('host');
  const metricAggregations = await (0, _query.getInventoryModelAggregations)('host', metrics.map(metric => metric), schema);
  const documentsFilter = await (0, _query.getDocumentsFilter)({
    apmDataAccessServices,
    apmDocumentSources,
    from,
    to,
    schema
  });
  const response = await infraMetricsClient.search({
    allow_no_indices: true,
    ignore_unavailable: true,
    size: 0,
    track_total_hits: false,
    query: {
      bool: {
        filter: [...(0, _server.termsQuery)(_constants.HOST_NAME_FIELD, ...hostNames), ...(0, _server.rangeQuery)(from, to)],
        should: [...documentsFilter]
      }
    },
    aggs: {
      // find hosts with metrics that are monitored by the system integration.
      monitoredHosts: {
        filter: {
          bool: {
            filter: [...((_inventoryModel$nodeF = (_inventoryModel$nodeF2 = inventoryModel.nodeFilter) === null || _inventoryModel$nodeF2 === void 0 ? void 0 : _inventoryModel$nodeF2.call(inventoryModel, {
              schema
            })) !== null && _inventoryModel$nodeF !== void 0 ? _inventoryModel$nodeF : [])]
          }
        },
        aggs: {
          names: {
            terms: {
              field: _constants.HOST_NAME_FIELD,
              size: limit,
              order: {
                _key: 'asc'
              }
            }
          }
        }
      },
      allHostMetrics: {
        terms: {
          field: _constants.HOST_NAME_FIELD,
          size: limit,
          order: {
            _key: 'asc'
          }
        },
        aggs: {
          ...metricAggregations,
          hostOsName: {
            filter: {
              exists: {
                field: 'host.os.name'
              }
            },
            aggs: {
              latest: {
                top_metrics: {
                  metrics: [{
                    field: 'host.os.name'
                  }],
                  size: 1,
                  sort: {
                    '@timestamp': 'desc'
                  }
                }
              }
            }
          },
          cloudProvider: {
            filter: {
              exists: {
                field: 'cloud.provider'
              }
            },
            aggs: {
              latest: {
                top_metrics: {
                  metrics: [{
                    field: 'cloud.provider'
                  }],
                  size: 1,
                  sort: {
                    '@timestamp': 'desc'
                  }
                }
              }
            }
          },
          hostIp: {
            filter: {
              exists: {
                field: 'host.ip'
              }
            },
            aggs: {
              latest: {
                top_metrics: {
                  metrics: [{
                    field: 'host.ip'
                  }],
                  size: 1,
                  sort: {
                    '@timestamp': 'desc'
                  }
                }
              }
            }
          }
        }
      }
    }
  });
  const systemIntegrationHosts = new Set((_response$aggregation = (_response$aggregation2 = response.aggregations) === null || _response$aggregation2 === void 0 ? void 0 : _response$aggregation2.monitoredHosts.names.buckets.map(p => p.key)) !== null && _response$aggregation !== void 0 ? _response$aggregation : []);
  const result = ((_response$aggregation3 = (_response$aggregation4 = response.aggregations) === null || _response$aggregation4 === void 0 ? void 0 : _response$aggregation4.allHostMetrics.buckets) !== null && _response$aggregation3 !== void 0 ? _response$aggregation3 : []).sort((a, b) => {
    var _getMetricValue, _getMetricValue2;
    const hasASystemMetrics = systemIntegrationHosts.has(a === null || a === void 0 ? void 0 : a.key);
    const hasBSystemMetrics = systemIntegrationHosts.has(b === null || b === void 0 ? void 0 : b.key);
    if (hasASystemMetrics !== hasBSystemMetrics) {
      return hasASystemMetrics ? -1 : 1;
    }
    const aValue = (_getMetricValue = getMetricValue(a === null || a === void 0 ? void 0 : a.cpuV2)) !== null && _getMetricValue !== void 0 ? _getMetricValue : 0;
    const bValue = (_getMetricValue2 = getMetricValue(b === null || b === void 0 ? void 0 : b.cpuV2)) !== null && _getMetricValue2 !== void 0 ? _getMetricValue2 : 0;
    return bValue - aValue;
  }).map(bucket => {
    var _bucket$hostOsName, _bucket$hostOsName$la, _bucket$hostOsName$la2, _bucket$hostOsName$la3, _bucket$hostOsName$la4, _bucket$cloudProvider, _bucket$cloudProvider2, _bucket$cloudProvider3, _bucket$cloudProvider4, _bucket$cloudProvider5, _bucket$hostIp, _bucket$hostIp$latest, _bucket$hostIp$latest2, _bucket$hostIp$latest3, _bucket$hostIp$latest4;
    const hostName = bucket.key;
    const hostOsNameValue = bucket === null || bucket === void 0 ? void 0 : (_bucket$hostOsName = bucket.hostOsName) === null || _bucket$hostOsName === void 0 ? void 0 : (_bucket$hostOsName$la = _bucket$hostOsName.latest) === null || _bucket$hostOsName$la === void 0 ? void 0 : (_bucket$hostOsName$la2 = _bucket$hostOsName$la.top) === null || _bucket$hostOsName$la2 === void 0 ? void 0 : (_bucket$hostOsName$la3 = _bucket$hostOsName$la2[0]) === null || _bucket$hostOsName$la3 === void 0 ? void 0 : (_bucket$hostOsName$la4 = _bucket$hostOsName$la3.metrics) === null || _bucket$hostOsName$la4 === void 0 ? void 0 : _bucket$hostOsName$la4['host.os.name'];
    const hostOsNameMetadata = [{
      name: 'host.os.name',
      value: hostOsNameValue !== null && hostOsNameValue !== void 0 ? hostOsNameValue : null
    }];
    const cloudProviderValue = bucket === null || bucket === void 0 ? void 0 : (_bucket$cloudProvider = bucket.cloudProvider) === null || _bucket$cloudProvider === void 0 ? void 0 : (_bucket$cloudProvider2 = _bucket$cloudProvider.latest) === null || _bucket$cloudProvider2 === void 0 ? void 0 : (_bucket$cloudProvider3 = _bucket$cloudProvider2.top) === null || _bucket$cloudProvider3 === void 0 ? void 0 : (_bucket$cloudProvider4 = _bucket$cloudProvider3[0]) === null || _bucket$cloudProvider4 === void 0 ? void 0 : (_bucket$cloudProvider5 = _bucket$cloudProvider4.metrics) === null || _bucket$cloudProvider5 === void 0 ? void 0 : _bucket$cloudProvider5['cloud.provider'];
    const cloudProviderMetadata = [{
      name: 'cloud.provider',
      value: cloudProviderValue !== null && cloudProviderValue !== void 0 ? cloudProviderValue : null
    }];
    const hostIpValue = bucket === null || bucket === void 0 ? void 0 : (_bucket$hostIp = bucket.hostIp) === null || _bucket$hostIp === void 0 ? void 0 : (_bucket$hostIp$latest = _bucket$hostIp.latest) === null || _bucket$hostIp$latest === void 0 ? void 0 : (_bucket$hostIp$latest2 = _bucket$hostIp$latest.top) === null || _bucket$hostIp$latest2 === void 0 ? void 0 : (_bucket$hostIp$latest3 = _bucket$hostIp$latest2[0]) === null || _bucket$hostIp$latest3 === void 0 ? void 0 : (_bucket$hostIp$latest4 = _bucket$hostIp$latest3.metrics) === null || _bucket$hostIp$latest4 === void 0 ? void 0 : _bucket$hostIp$latest4['host.ip'];
    const hostIpMetadata = [{
      name: 'host.ip',
      value: hostIpValue !== null && hostIpValue !== void 0 ? hostIpValue : null
    }];
    const metadata = [...hostOsNameMetadata, ...cloudProviderMetadata, ...hostIpMetadata];
    return {
      name: hostName,
      metadata,
      metrics: metrics.map(metric => ({
        name: metric,
        value: getMetricValue(bucket[metric]) || null
      })),
      hasSystemMetrics: systemIntegrationHosts.has(hostName)
    };
  });
  return result;
};
exports.getAllHosts = getAllHosts;
const getMetricValue = valueObject => {
  if (_server2.BasicMetricValueRT.is(valueObject)) {
    return valueObject.value;
  }
  return valueObject;
};