"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.parseThemeNameValue = exports.hasNonDefaultThemeTags = exports.ThemeBorealisTags = exports.ThemeAmsterdamTags = exports.SUPPORTED_THEME_TAGS = exports.SUPPORTED_THEME_NAMES = exports.FALLBACK_THEME_TAG = exports.DEFAULT_THEME_TAGS = exports.DEFAULT_THEME_NAME = void 0;
exports.parseThemeTags = parseThemeTags;
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const DEFAULT_THEME_NAME = exports.DEFAULT_THEME_NAME = 'borealis';
const SUPPORTED_THEME_NAMES = exports.SUPPORTED_THEME_NAMES = ['amsterdam', 'borealis'];
/**
 * Theme tags of the Amsterdam theme
 */
const ThemeAmsterdamTags = exports.ThemeAmsterdamTags = ['v8light', 'v8dark'];

/**
 * Theme tags of the experimental Borealis theme
 */
const ThemeBorealisTags = exports.ThemeBorealisTags = ['borealislight', 'borealisdark'];

/**
 * An array of all theme tags supported by Kibana. Note that this list doesn't
 * reflect what theme tags are available in a Kibana build.
 */
const SUPPORTED_THEME_TAGS = exports.SUPPORTED_THEME_TAGS = [...ThemeAmsterdamTags, ...ThemeBorealisTags];
/**
 * An array of theme tags available in Kibana by default when not customized
 * using KBN_OPTIMIZER_THEMES environment variable.
 */
const DEFAULT_THEME_TAGS = exports.DEFAULT_THEME_TAGS = ThemeBorealisTags;
const FALLBACK_THEME_TAG = exports.FALLBACK_THEME_TAG = 'borealislight';
const isValidTag = tag => SUPPORTED_THEME_TAGS.includes(tag);
function parseThemeTags(input) {
  if (!input || input === '*') {
    return DEFAULT_THEME_TAGS;
  }
  let rawTags;
  if (typeof input === 'string') {
    rawTags = input.split(',').map(tag => tag.trim());
  } else if (Array.isArray(input)) {
    rawTags = input;
  } else {
    throw new Error('Invalid theme tags, must be an array of strings');
  }
  if (!rawTags.length) {
    throw new Error(`Invalid theme tags, you must specify at least one of [${SUPPORTED_THEME_TAGS.join(', ')}]`);
  }
  const invalidTags = rawTags.filter(t => !isValidTag(t));
  if (invalidTags.length) {
    throw new Error(`Invalid theme tags [${invalidTags.join(', ')}], options: [${SUPPORTED_THEME_TAGS.join(', ')}]`);
  }
  return rawTags;
}
const hasNonDefaultThemeTags = tags => tags.length !== DEFAULT_THEME_TAGS.length || tags.some(tag => !DEFAULT_THEME_TAGS.includes(tag));
exports.hasNonDefaultThemeTags = hasNonDefaultThemeTags;
const parseThemeNameValue = value => {
  if (typeof value !== 'string') {
    return DEFAULT_THEME_NAME;
  }
  const themeName = value.toLowerCase();
  if (SUPPORTED_THEME_NAMES.includes(themeName.toLowerCase())) {
    return themeName;
  }
  return DEFAULT_THEME_NAME;
};
exports.parseThemeNameValue = parseThemeNameValue;