"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.alwaysDisplayedFields = void 0;
exports.getEventCategoriesFromData = getEventCategoriesFromData;
exports.getHighlightedFieldsToDisplay = getHighlightedFieldsToDisplay;
var _fp = require("lodash/fp");
var _ruleDataUtils = require("@kbn/rule-data-utils");
var _securitysolutionEcs = require("@kbn/securitysolution-ecs");
var _i18n = require("@kbn/i18n");
var _constants = require("../../../../common/endpoint/service/response_actions/constants");
var _translations = require("../../../detections/components/alerts_table/translations");
var _field_names = require("../../../../common/field_maps/field_names");
var _constants2 = require("../../../timelines/components/timeline/body/renderers/constants");
var _translations2 = require("./translations");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const THRESHOLD_TERMS_FIELD = `${_field_names.ALERT_THRESHOLD_RESULT}.terms.field`;
const THRESHOLD_TERMS_VALUE = `${_field_names.ALERT_THRESHOLD_RESULT}.terms.value`;
const THRESHOLD_CARDINALITY_FIELD = `${_field_names.ALERT_THRESHOLD_RESULT}.cardinality.field`;
const THRESHOLD_COUNT = `${_field_names.ALERT_THRESHOLD_RESULT}.count`;
const AGENT_STATUS = _i18n.i18n.translate('xpack.securitySolution.detections.alerts.agentStatus', {
  defaultMessage: 'Agent status'
});
const QUARANTINED_FILE_PATH = _i18n.i18n.translate('xpack.securitySolution.detections.alerts.quarantinedFilePath', {
  defaultMessage: 'Quarantined file path'
});
const RULE_TYPE = _i18n.i18n.translate('xpack.securitySolution.detections.alerts.ruleType', {
  defaultMessage: 'Rule type'
});

/** Always show these fields */
const alwaysDisplayedFields = exports.alwaysDisplayedFields = [{
  id: _constants2.EVENT_SOURCE_FIELD_NAME,
  overrideField: _translations2.EVENT_SOURCE_FIELD_DESCRIPTOR
}, {
  id: 'host.name'
},
// Add all fields used to identify the agent ID in alert events and override them to
// show the `agent.status` field name/value
..._constants.SUPPORTED_AGENT_ID_ALERT_FIELDS.map(fieldPath => {
  return {
    id: fieldPath,
    overrideField: _constants2.AGENT_STATUS_FIELD_NAME,
    label: AGENT_STATUS
  };
}),
// ** //
{
  id: 'Endpoint.policy.applied.artifacts.global.channel'
}, {
  id: 'user.name'
}, {
  id: 'rule.name'
}, {
  id: 'cloud.provider'
}, {
  id: 'cloud.region'
}, {
  id: 'orchestrator.cluster.id'
}, {
  id: 'orchestrator.cluster.name'
}, {
  id: 'container.image.name'
}, {
  id: 'container.image.tag'
}, {
  id: 'orchestrator.namespace'
}, {
  id: 'orchestrator.resource.parent.type'
}, {
  id: 'orchestrator.resource.type'
}, {
  id: 'process.executable'
}, {
  id: 'file.path'
}, {
  id: _ruleDataUtils.ALERT_RULE_TYPE,
  label: RULE_TYPE
}];

/**
 * Get a list of fields to display based on the event's category
 */
function getFieldsByCategory({
  primaryEventCategory,
  allEventCategories
}) {
  switch (primaryEventCategory) {
    case _securitysolutionEcs.EventCategory.PROCESS:
      return [{
        id: 'process.name'
      }, {
        id: 'process.parent.name'
      }, {
        id: 'process.args'
      }];
    case _securitysolutionEcs.EventCategory.FILE:
      return [{
        id: 'file.name'
      }, {
        id: 'file.hash.sha256'
      }, {
        id: 'file.directory'
      }, {
        id: 'process.name'
      }];
    case _securitysolutionEcs.EventCategory.NETWORK:
      return [{
        id: 'destination.address'
      }, {
        id: 'destination.port'
      }, {
        id: 'source.address'
      }, {
        id: 'source.port'
      }, {
        id: 'dns.question.name'
      }, {
        id: 'process.name'
      }];
    case _securitysolutionEcs.EventCategory.REGISTRY:
      return [{
        id: 'registry.key'
      }, {
        id: 'registry.value'
      }, {
        id: 'process.name'
      }];
    case _securitysolutionEcs.EventCategory.MALWARE:
      // The details for malware events can be found in the file fields
      return getFieldsByCategory({
        primaryEventCategory: _securitysolutionEcs.EventCategory.FILE,
        allEventCategories
      });
    default:
      let fields = [];

      // If no primary category matches or hasn't been defined on purpose (e.g. in order to follow the source event)
      // resolve more fields based on the other event categories.
      if (allEventCategories !== null && allEventCategories !== void 0 && allEventCategories.includes(_securitysolutionEcs.EventCategory.FILE)) {
        fields = fields.concat(getFieldsByCategory({
          primaryEventCategory: _securitysolutionEcs.EventCategory.FILE
        }));
      }
      if (allEventCategories !== null && allEventCategories !== void 0 && allEventCategories.includes(_securitysolutionEcs.EventCategory.PROCESS)) {
        fields = fields.concat(getFieldsByCategory({
          primaryEventCategory: _securitysolutionEcs.EventCategory.PROCESS
        }));
      }
      return fields;
  }
}

/**
 * Gets the fields to display based on the event's code.
 * Contains some enhancements to resolve more fields based on the event's categories.
 * @param eventCode The event's code
 * @param eventCategories The events categories
 * @returns A list of fields to include
 */
function getFieldsByEventCode(eventCode, eventCategories) {
  switch (eventCode) {
    case _securitysolutionEcs.EventCode.BEHAVIOR:
      return [{
        id: 'rule.description',
        label: _translations.ALERTS_HEADERS_RULE_DESCRIPTION
      },
      // Resolve more fields based on the source event
      ...getFieldsByCategory({
        ...eventCategories,
        primaryEventCategory: undefined
      })];
    case _securitysolutionEcs.EventCode.SHELLCODE_THREAD:
      return [{
        id: 'Target.process.executable'
      }, {
        id: 'Memory_protection.unique_key_v1'
      }];
    case _securitysolutionEcs.EventCode.RANSOMWARE:
      return [{
        id: 'Ransomware.feature'
      }, {
        id: 'process.hash.sha256'
      }, ...getFieldsByCategory({
        ...eventCategories,
        primaryEventCategory: undefined
      })];
    case _securitysolutionEcs.EventCode.MEMORY_SIGNATURE:
      // Resolve more fields based on the source event
      return getFieldsByCategory({
        ...eventCategories,
        primaryEventCategory: undefined
      });
    case _securitysolutionEcs.EventCode.MALICIOUS_FILE:
      return [{
        id: 'file.Ext.quarantine_path',
        overrideField: _constants2.QUARANTINED_PATH_FIELD_NAME,
        label: QUARANTINED_FILE_PATH
      }];
    default:
      return [];
  }
}

/**
 * Returns a list of fields based on the event's rule type
 */
function getFieldsByRuleType(ruleType) {
  switch (ruleType) {
    case 'threshold':
      return [{
        id: THRESHOLD_COUNT,
        label: _translations.ALERTS_HEADERS_THRESHOLD_COUNT
      }, {
        id: THRESHOLD_TERMS_FIELD,
        overrideField: THRESHOLD_TERMS_VALUE,
        label: _translations.ALERTS_HEADERS_THRESHOLD_TERMS
      }, {
        id: THRESHOLD_CARDINALITY_FIELD,
        label: _translations.ALERTS_HEADERS_THRESHOLD_CARDINALITY
      }];
    case 'machine_learning':
      return [{
        id: `${_ruleDataUtils.ALERT_RULE_PARAMETERS}.machine_learning_job_id`,
        legacyId: 'signal.rule.machine_learning_job_id'
      }, {
        id: `${_ruleDataUtils.ALERT_RULE_PARAMETERS}.anomaly_threshold`,
        legacyId: 'signal.rule.anomaly_threshold'
      }];
    case 'threat_match':
      return [{
        id: `${_ruleDataUtils.ALERT_RULE_PARAMETERS}.threat_index`,
        legacyId: 'signal.rule.threat_index'
      }, {
        id: `${_ruleDataUtils.ALERT_RULE_PARAMETERS}.threat_query`,
        legacyId: 'signal.rule.threat_query'
      }];
    case 'new_terms':
      return [{
        id: _field_names.ALERT_NEW_TERMS_FIELDS,
        label: _translations.ALERTS_HEADERS_NEW_TERMS_FIELDS
      }, {
        id: _field_names.ALERT_NEW_TERMS,
        label: _translations.ALERTS_HEADERS_NEW_TERMS
      }];
    default:
      return [];
  }
}

/**
 * Gets the fields to display based on custom rules and configuration
 * @param customs The list of custom-defined fields to display
 * @returns The list of custom-defined fields to display
 */
function getCustomHighlightedFields(customs) {
  return customs.map(field => ({
    id: field
  }));
}

/**
  This function is exported because it is used in the Exception Component to
  populate the conditions with the Highlighted Fields. Additionally, the new
  Alert Summary Flyout also requires access to these fields.
  As the Alert Summary components will undergo changes soon we will go with
  exporting the function only for now.
 */
/**
 * Assembles a list of fields to display based on the event
 */
function getHighlightedFieldsToDisplay({
  eventCategories,
  eventCode,
  eventRuleType,
  ruleCustomHighlightedFields,
  type = 'all'
}) {
  const customHighlightedFields = getCustomHighlightedFields(ruleCustomHighlightedFields);
  const defaultHighlightedFields = [...alwaysDisplayedFields, ...getFieldsByCategory(eventCategories), ...getFieldsByEventCode(eventCode, eventCategories), ...getFieldsByRuleType(eventRuleType)];
  if (type === 'default') {
    return (0, _fp.uniqBy)('id', defaultHighlightedFields);
  }
  if (type === 'custom') {
    return customHighlightedFields;
  }
  return (0, _fp.uniqBy)('id', [...customHighlightedFields, ...defaultHighlightedFields]);
}
/**
 * Extract the event's categories
 * @param data The event details
 * @returns The event's primary category and all other categories in case there is more than one
 */
function getEventCategoriesFromData(data) {
  const eventCategoryField = (0, _fp.find)({
    category: 'event',
    field: 'event.category'
  }, data);
  let primaryEventCategory;
  let allEventCategories;
  if (Array.isArray(eventCategoryField === null || eventCategoryField === void 0 ? void 0 : eventCategoryField.originalValue)) {
    primaryEventCategory = eventCategoryField === null || eventCategoryField === void 0 ? void 0 : eventCategoryField.originalValue[0];
    allEventCategories = eventCategoryField === null || eventCategoryField === void 0 ? void 0 : eventCategoryField.originalValue;
  } else {
    primaryEventCategory = eventCategoryField === null || eventCategoryField === void 0 ? void 0 : eventCategoryField.originalValue;
    if (primaryEventCategory) {
      allEventCategories = [primaryEventCategory];
    }
  }
  return {
    primaryEventCategory,
    allEventCategories
  };
}