"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createToolRegistry = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _onechatCommon = require("@kbn/onechat-common");
var _tool_provider = require("./tool_provider");
var _utils = require("./utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

// eslint-disable-next-line @typescript-eslint/no-empty-interface

const createToolRegistry = params => {
  return new ToolRegistryImpl(params);
};
exports.createToolRegistry = createToolRegistry;
class ToolRegistryImpl {
  constructor({
    persistedProvider,
    builtinProvider,
    request,
    getRunner
  }) {
    (0, _defineProperty2.default)(this, "persistedProvider", void 0);
    (0, _defineProperty2.default)(this, "builtinProvider", void 0);
    (0, _defineProperty2.default)(this, "request", void 0);
    (0, _defineProperty2.default)(this, "getRunner", void 0);
    this.persistedProvider = persistedProvider;
    this.builtinProvider = builtinProvider;
    this.request = request;
    this.getRunner = getRunner;
  }
  get orderedProviders() {
    return [this.builtinProvider, this.persistedProvider];
  }
  async execute(params) {
    const {
      toolId,
      ...otherParams
    } = params;
    const tool = await this.get(toolId);
    const executable = (0, _utils.toExecutableTool)({
      tool,
      runner: this.getRunner(),
      request: this.request
    });
    return await executable.execute(otherParams);
  }
  async has(toolId) {
    for (const provider of this.orderedProviders) {
      if (await provider.has(toolId)) {
        return true;
      }
    }
    return false;
  }
  async get(toolId) {
    for (const provider of this.orderedProviders) {
      if (await provider.has(toolId)) {
        return provider.get(toolId);
      }
    }
    throw (0, _onechatCommon.createToolNotFoundError)({
      toolId
    });
  }
  async list(opts) {
    const allTools = [];
    for (const provider of this.orderedProviders) {
      const toolsFromType = await provider.list();
      allTools.push(...toolsFromType);
    }
    return allTools;
  }
  async create(createRequest) {
    const {
      id: toolId
    } = createRequest;
    const validationError = (0, _onechatCommon.validateToolId)({
      toolId,
      builtIn: false
    });
    if (validationError) {
      throw (0, _onechatCommon.createBadRequestError)(`Invalid tool id: "${toolId}": ${validationError}`);
    }
    if (await this.has(toolId)) {
      throw (0, _onechatCommon.createBadRequestError)(`Tool with id ${toolId} already exists`);
    }
    return this.persistedProvider.create(createRequest);
  }
  async update(toolId, update) {
    for (const provider of this.orderedProviders) {
      if (await provider.has(toolId)) {
        if ((0, _tool_provider.isReadonlyToolProvider)(provider)) {
          throw (0, _onechatCommon.createBadRequestError)(`Tool ${toolId} is read-only and can't be updated`);
        }
        return provider.update(toolId, update);
      }
    }
    throw (0, _onechatCommon.createToolNotFoundError)({
      toolId
    });
  }
  async delete(toolId) {
    for (const provider of this.orderedProviders) {
      if (await provider.has(toolId)) {
        if ((0, _tool_provider.isReadonlyToolProvider)(provider)) {
          throw (0, _onechatCommon.createBadRequestError)(`Tool ${toolId} is read-only and can't be deleted`);
        }
        return provider.delete(toolId);
      }
    }
    throw (0, _onechatCommon.createToolNotFoundError)({
      toolId
    });
  }
}