"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getInfluencerValueMaxScoreByTime = getInfluencerValueMaxScoreByTime;
exports.getScoresByBucket = getScoresByBucket;
var _explorer = require("../../../common/constants/explorer");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

async function getScoresByBucket(mlClient, {
  jobIds,
  earliestMs,
  latestMs,
  intervalMs,
  perPage = 10,
  fromPage = 1,
  swimLaneSeverity = [{
    min: 0
  }]
}) {
  var _jobIds$length, _jobId$buckets, _resp$aggregations, _resp$aggregations$jo, _jobsCardinality$valu, _resp$aggregations2, _resp$aggregations2$j;
  const boolCriteria = [{
    range: {
      timestamp: {
        gte: earliestMs,
        lte: latestMs,
        format: 'epoch_millis'
      }
    }
  }];
  const thresholdCriteria = swimLaneSeverity.map(t => ({
    range: {
      anomaly_score: {
        gte: t.min,
        ...(t.max !== undefined ? {
          lte: t.max
        } : {})
      }
    }
  }));
  boolCriteria.push({
    bool: {
      should: thresholdCriteria,
      minimum_should_match: 1
    }
  });
  if (jobIds && jobIds.length > 0 && !(jobIds.length === 1 && jobIds[0] === '*')) {
    const query = jobIds.map(j => `job_id:${j}`).join(' OR ');
    boolCriteria.push({
      query_string: {
        analyze_wildcard: false,
        query
      }
    });
  }
  const resp = await mlClient.anomalySearch({
    size: 0,
    query: {
      bool: {
        filter: [{
          query_string: {
            query: 'result_type:bucket',
            analyze_wildcard: false
          }
        }, {
          bool: {
            must: boolCriteria
          }
        }]
      }
    },
    aggs: {
      jobsCardinality: {
        cardinality: {
          field: 'job_id'
        }
      },
      jobId: {
        terms: {
          field: 'job_id',
          size: (_jobIds$length = jobIds === null || jobIds === void 0 ? void 0 : jobIds.length) !== null && _jobIds$length !== void 0 ? _jobIds$length : 1,
          order: {
            anomalyScore: 'desc'
          }
        },
        aggs: {
          anomalyScore: {
            max: {
              field: 'anomaly_score'
            }
          },
          bucketTruncate: {
            bucket_sort: {
              from: (fromPage - 1) * perPage,
              size: perPage === 0 ? 1 : perPage
            }
          },
          byTime: {
            date_histogram: {
              field: 'timestamp',
              fixed_interval: `${intervalMs}ms`,
              min_doc_count: 1,
              extended_bounds: {
                min: earliestMs,
                max: latestMs
              }
            },
            aggs: {
              anomalyScore: {
                max: {
                  field: 'anomaly_score'
                }
              }
            }
          }
        }
      }
    }
  }, jobIds);
  const results = {};
  const buckets = (_jobId$buckets = (_resp$aggregations = resp.aggregations) === null || _resp$aggregations === void 0 ? void 0 : (_resp$aggregations$jo = _resp$aggregations.jobId) === null || _resp$aggregations$jo === void 0 ? void 0 : _resp$aggregations$jo.buckets) !== null && _jobId$buckets !== void 0 ? _jobId$buckets : [];
  for (const jobBucket of buckets) {
    var _jobBucket$byTime$buc, _jobBucket$byTime;
    const jobId = jobBucket.key;
    const byTime = (_jobBucket$byTime$buc = (_jobBucket$byTime = jobBucket.byTime) === null || _jobBucket$byTime === void 0 ? void 0 : _jobBucket$byTime.buckets) !== null && _jobBucket$byTime$buc !== void 0 ? _jobBucket$byTime$buc : [];
    const resultForTime = {};
    for (const t of byTime) {
      var _t$anomalyScore;
      const value = (_t$anomalyScore = t.anomalyScore) === null || _t$anomalyScore === void 0 ? void 0 : _t$anomalyScore.value;
      if (value !== undefined) {
        resultForTime[t.key] = value;
      }
    }
    results[jobId] = resultForTime;
  }
  const cardinality = (_jobsCardinality$valu = (_resp$aggregations2 = resp.aggregations) === null || _resp$aggregations2 === void 0 ? void 0 : (_resp$aggregations2$j = _resp$aggregations2.jobsCardinality) === null || _resp$aggregations2$j === void 0 ? void 0 : _resp$aggregations2$j.value) !== null && _jobsCardinality$valu !== void 0 ? _jobsCardinality$valu : 0;
  return {
    results,
    cardinality
  };
}
async function getInfluencerValueMaxScoreByTime(mlClient, {
  jobIds,
  influencerFieldName,
  influencerFieldValues = [],
  earliestMs,
  latestMs,
  intervalMs,
  maxResults,
  perPage = 10,
  fromPage = 1,
  influencersFilterQuery,
  swimLaneSeverity = [{
    min: 0
  }]
}) {
  var _influencerFieldValue, _resp$aggregations3, _resp$aggregations3$i, _influencerValuesCard, _resp$aggregations4, _resp$aggregations4$i;
  const boolCriteria = [{
    range: {
      timestamp: {
        gte: earliestMs,
        lte: latestMs,
        format: 'epoch_millis'
      }
    }
  }];
  const thresholdCriteria = swimLaneSeverity.map(t => ({
    range: {
      influencer_score: {
        gte: t.min,
        ...(t.max !== undefined ? {
          lte: t.max
        } : {})
      }
    }
  }));
  boolCriteria.push({
    bool: {
      should: thresholdCriteria,
      minimum_should_match: 1
    }
  });
  if (jobIds && jobIds.length > 0 && !(jobIds.length === 1 && jobIds[0] === '*')) {
    const query = jobIds.map(j => `job_id:${j}`).join(' OR ');
    boolCriteria.push({
      query_string: {
        analyze_wildcard: false,
        query
      }
    });
  }
  if (influencersFilterQuery !== undefined) {
    boolCriteria.push(influencersFilterQuery);
  }
  if (influencerFieldValues.length > 0) {
    boolCriteria.push({
      bool: {
        should: influencerFieldValues.map(v => ({
          term: {
            influencer_field_value: v
          }
        })),
        minimum_should_match: 1
      }
    });
  }
  const resp = await mlClient.anomalySearch({
    size: 0,
    query: {
      bool: {
        filter: [{
          query_string: {
            query: 'result_type:influencer',
            analyze_wildcard: false
          }
        }, {
          term: {
            influencer_field_name: influencerFieldName
          }
        }, {
          bool: {
            must: boolCriteria
          }
        }]
      }
    },
    aggs: {
      influencerValuesCardinality: {
        cardinality: {
          field: 'influencer_field_value'
        }
      },
      influencerFieldValues: {
        terms: {
          field: 'influencer_field_value',
          size: !!maxResults ? maxResults : _explorer.ANOMALY_SWIM_LANE_HARD_LIMIT,
          order: {
            maxAnomalyScore: 'desc'
          }
        },
        aggs: {
          maxAnomalyScore: {
            max: {
              field: 'influencer_score'
            }
          },
          bucketTruncate: {
            bucket_sort: {
              from: (fromPage - 1) * perPage,
              size: perPage
            }
          },
          byTime: {
            date_histogram: {
              field: 'timestamp',
              fixed_interval: `${intervalMs}ms`,
              min_doc_count: 1
            },
            aggs: {
              maxAnomalyScore: {
                max: {
                  field: 'influencer_score'
                }
              }
            }
          }
        }
      }
    }
  }, jobIds);
  const results = {};
  const valueBuckets = (_influencerFieldValue = (_resp$aggregations3 = resp.aggregations) === null || _resp$aggregations3 === void 0 ? void 0 : (_resp$aggregations3$i = _resp$aggregations3.influencerFieldValues) === null || _resp$aggregations3$i === void 0 ? void 0 : _resp$aggregations3$i.buckets) !== null && _influencerFieldValue !== void 0 ? _influencerFieldValue : [];
  for (const valueBucket of valueBuckets) {
    var _valueBucket$byTime$b, _valueBucket$byTime;
    const fieldValue = valueBucket.key;
    const fieldValues = {};
    const timeBuckets = (_valueBucket$byTime$b = (_valueBucket$byTime = valueBucket.byTime) === null || _valueBucket$byTime === void 0 ? void 0 : _valueBucket$byTime.buckets) !== null && _valueBucket$byTime$b !== void 0 ? _valueBucket$byTime$b : [];
    for (const t of timeBuckets) {
      var _t$maxAnomalyScore$va, _t$maxAnomalyScore;
      fieldValues[t.key] = (_t$maxAnomalyScore$va = (_t$maxAnomalyScore = t.maxAnomalyScore) === null || _t$maxAnomalyScore === void 0 ? void 0 : _t$maxAnomalyScore.value) !== null && _t$maxAnomalyScore$va !== void 0 ? _t$maxAnomalyScore$va : 0;
    }
    results[fieldValue] = fieldValues;
  }
  const cardinality = (_influencerValuesCard = (_resp$aggregations4 = resp.aggregations) === null || _resp$aggregations4 === void 0 ? void 0 : (_resp$aggregations4$i = _resp$aggregations4.influencerValuesCardinality) === null || _resp$aggregations4$i === void 0 ? void 0 : _resp$aggregations4$i.value) !== null && _influencerValuesCard !== void 0 ? _influencerValuesCard : 0;
  return {
    results,
    cardinality
  };
}