"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getMetricFields = getMetricFields;
var _lodash = require("lodash");
var _deduplicate_fields = require("../../lib/fields/deduplicate_fields");
var _extract_metric_fields = require("../../lib/fields/extract_metric_fields");
var _enrich_metric_fields = require("../../lib/fields/enrich_metric_fields");
var _extract_dimensions = require("../../lib/dimensions/extract_dimensions");
var _build_metric_field = require("../../lib/fields/build_metric_field");
var _retrieve_fieldcaps = require("../../lib/fields/retrieve_fieldcaps");
var _apply_pagination = require("../../lib/pagination/apply_pagination");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

async function getMetricFields({
  indexPattern,
  fields = '*',
  esClient,
  page,
  size,
  logger,
  timerange
}) {
  if (!indexPattern) return {
    fields: [],
    total: 0
  };
  const indexFieldCapsMap = await (0, _retrieve_fieldcaps.retrieveFieldCaps)({
    esClient: esClient.client,
    indexPattern,
    fields,
    timerange
  });
  const allMetricFields = [];
  for (const [indexName, fieldCaps] of indexFieldCapsMap.entries()) {
    if ((0, _lodash.isNumber)(indexName) || fieldCaps == null) continue;
    if (Object.keys(fieldCaps).length === 0) continue;
    const metricFields = (0, _extract_metric_fields.extractMetricFields)(fieldCaps);
    const allDimensions = (0, _extract_dimensions.extractDimensions)(fieldCaps);
    const initialFields = metricFields.map(({
      fieldName,
      type,
      typeInfo
    }) => (0, _build_metric_field.buildMetricField)({
      name: fieldName,
      index: indexName,
      dimensions: allDimensions,
      type,
      typeInfo
    }));
    const deduped = (0, _deduplicate_fields.deduplicateFields)(initialFields);
    allMetricFields.push(...deduped);
  }
  allMetricFields.sort((a, b) => (0, _enrich_metric_fields.generateMapKey)(a.index, a.name).localeCompare((0, _enrich_metric_fields.generateMapKey)(b.index, b.name)));
  const enrichedMetricFields = await (0, _enrich_metric_fields.enrichMetricFields)({
    esClient,
    metricFields: (0, _apply_pagination.applyPagination)({
      metricFields: allMetricFields,
      page,
      size
    }),
    indexFieldCapsMap,
    logger,
    timerange
  });
  const finalFields = enrichedMetricFields.map(field => {
    return {
      ...field,
      dimensions: field.dimensions.sort((a, b) => a.name.localeCompare(b.name))
    };
  });
  return {
    fields: finalFields,
    total: allMetricFields.length
  };
}