"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerLensVisualizationsCreateAPIRoute = void 0;
var _lodash = require("lodash");
var _boom = require("@hapi/boom");
var _constants = require("../../../../common/constants");
var _transforms = require("../../../../common/transforms");
var _schema = require("./schema");
var _utils = require("../utils");
var _config_builder_stub = require("../../../../common/transforms/config_builder_stub");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const registerLensVisualizationsCreateAPIRoute = (router, {
  contentManagement
}) => {
  const createRoute = router.post({
    path: _constants.LENS_VIS_API_PATH,
    access: _constants.LENS_API_ACCESS,
    enableQueryVersion: true,
    summary: 'Create Lens visualization',
    description: 'Create a new Lens visualization.',
    options: {
      tags: ['oas-tag:Lens'],
      availability: {
        stability: 'experimental'
      }
    },
    security: {
      authz: {
        enabled: false,
        reason: 'Relies on Content Client for authorization'
      }
    }
  });
  createRoute.addVersion({
    version: _constants.LENS_API_VERSION,
    validate: {
      request: {
        body: _schema.lensCreateRequestBodySchema
      },
      response: {
        201: {
          body: () => _schema.lensCreateResponseBodySchema,
          description: 'Created'
        },
        400: {
          description: 'Malformed request'
        },
        401: {
          description: 'Unauthorized'
        },
        403: {
          description: 'Forbidden'
        },
        500: {
          description: 'Internal Server Error'
        }
      }
    }
  }, async (ctx, req, res) => {
    var _requestBodyData$desc;
    const requestBodyData = req.body.data;
    if (!requestBodyData.visualizationType) {
      throw new Error('visualizationType is required');
    }

    // TODO fix IContentClient to type this client based on the actual
    const client = contentManagement.contentClient.getForRequest({
      request: req,
      requestHandlerContext: ctx
    }).for(_constants.LENS_CONTENT_TYPE);
    const {
      references,
      ...lensItem
    } = (0, _config_builder_stub.isNewApiFormat)(requestBodyData) ?
    // TODO: Find a better way to conditionally omit id
    (0, _lodash.omit)(_transforms.ConfigBuilderStub.in(requestBodyData), 'id') :
    // For now we need to be able to create old SO, this may be moved to the config builder
    {
      ...requestBodyData,
      // fix type mismatches, null -> undefined
      description: (_requestBodyData$desc = requestBodyData.description) !== null && _requestBodyData$desc !== void 0 ? _requestBodyData$desc : undefined,
      visualizationType: requestBodyData.visualizationType
    };
    try {
      // Note: these types are to enforce loose param typings of client methods
      const data = lensItem;
      const options = {
        ...req.body.options,
        references
      };
      const {
        result
      } = await client.create(data, options);
      if (result.item.error) {
        throw result.item.error;
      }
      return res.created({
        body: (0, _utils.getLensResponseItem)(result.item)
      });
    } catch (error) {
      if ((0, _boom.isBoom)(error) && error.output.statusCode === 403) {
        return res.forbidden();
      }
      return (0, _boom.boomify)(error); // forward unknown error
    }
  });
};
exports.registerLensVisualizationsCreateAPIRoute = registerLensVisualizationsCreateAPIRoute;