"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.autocomplete = autocomplete;
var _esqlTypes = require("@kbn/esql-types");
var _walker = require("../../../walker");
var _functions = require("../../../definitions/utils/autocomplete/functions");
var _is = require("../../../ast/is");
var _types = require("../../types");
var _complete_items = require("../../complete_items");
var _helpers = require("../../../definitions/utils/autocomplete/helpers");
var _expressions = require("../../../definitions/utils/expressions");
var _constants = require("../../constants");
var _utils = require("./utils");
var _ast = require("../../../definitions/utils/ast");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

function alreadyUsedColumns(command) {
  var _byOption$args$filter;
  const byOption = command.args.find(arg => !Array.isArray(arg) && arg.name === 'by');
  const columnNodes = (_byOption$args$filter = byOption === null || byOption === void 0 ? void 0 : byOption.args.filter(arg => !Array.isArray(arg) && arg.type === 'column')) !== null && _byOption$args$filter !== void 0 ? _byOption$args$filter : [];
  return columnNodes.map(node => node.parts.join('.'));
}
async function autocomplete(query, command, callbacks, context, cursorPosition) {
  if (!(callbacks !== null && callbacks !== void 0 && callbacks.getByType)) {
    return [];
  }
  const isInlineStats = command.name === 'inline stats';
  const columnExists = name => (0, _helpers.columnExists)(name, context);
  const innerText = query.substring(0, cursorPosition);
  const pos = (0, _utils.getPosition)(command, innerText);
  const lastCharacterTyped = innerText[innerText.length - 1];
  const controlSuggestions = (0, _helpers.getControlSuggestionIfSupported)(Boolean(context === null || context === void 0 ? void 0 : context.supportsControls), _esqlTypes.ESQLVariableType.FUNCTIONS, context === null || context === void 0 ? void 0 : context.variables, lastCharacterTyped !== _constants.ESQL_VARIABLES_PREFIX);
  const functionsSpecificSuggestions = await (0, _functions.getInsideFunctionsSuggestions)(query, cursorPosition, callbacks, context);
  if (functionsSpecificSuggestions && cursorPosition && _walker.Walker.findFunction(command, fn => (0, _helpers.within)(cursorPosition, fn.location))) {
    return functionsSpecificSuggestions;
  }
  switch (pos) {
    case 'expression_without_assignment':
      {
        var _callbacks$getSuggest;
        const isNewMultipleExpression = /,\s*$/.test(innerText);
        const expressionRoot = isNewMultipleExpression ? undefined // we're in a new expression, but there isn't an AST node for it yet
        : command.args[command.args.length - 1];
        if (Array.isArray(expressionRoot)) {
          return [];
        }
        const expressionSuggestions = await getExpressionSuggestions({
          innerText,
          expressionRoot,
          location: _types.Location.STATS,
          context,
          callbacks,
          emptySuggestions: [...(!isNewMultipleExpression && !isInlineStats ? [{
            ..._complete_items.byCompleteItem,
            sortText: 'D'
          }] : []), (0, _complete_items.getNewUserDefinedColumnSuggestion)((callbacks === null || callbacks === void 0 ? void 0 : (_callbacks$getSuggest = callbacks.getSuggestedUserDefinedColumnName) === null || _callbacks$getSuggest === void 0 ? void 0 : _callbacks$getSuggest.call(callbacks)) || '')],
          afterCompleteSuggestions: [_complete_items.whereCompleteItem, _complete_items.byCompleteItem, ...(0, _utils.getCommaAndPipe)(innerText, expressionRoot, columnExists)],
          suggestColumns: false
        });
        return [...expressionSuggestions, ...controlSuggestions];
      }
    case 'expression_after_assignment':
      {
        // Find expression root
        const assignment = command.args[command.args.length - 1];
        const rightHandAssignment = (0, _is.isAssignment)(assignment) ? assignment.args[assignment.args.length - 1] : undefined;
        let expressionRoot = Array.isArray(rightHandAssignment) ? rightHandAssignment[0] : undefined;

        // @TODO the marker shouldn't be leaking through here
        if ((0, _ast.isMarkerNode)(expressionRoot)) {
          expressionRoot = undefined;
        }
        if (Array.isArray(expressionRoot)) {
          return [];
        }
        const expressionSuggestions = await getExpressionSuggestions({
          innerText,
          expressionRoot,
          location: _types.Location.STATS,
          context,
          callbacks,
          emptySuggestions: [],
          afterCompleteSuggestions: [_complete_items.whereCompleteItem, _complete_items.byCompleteItem, ...(0, _utils.getCommaAndPipe)(innerText, expressionRoot, columnExists)],
          suggestColumns: false
        });
        return [...expressionSuggestions, ...controlSuggestions];
      }
    case 'after_where':
      {
        const whereFn = command.args[command.args.length - 1];
        // TODO do we still need this check?
        const expressionRoot = (0, _ast.isMarkerNode)(whereFn.args[1]) ? undefined : whereFn.args[1];
        if (expressionRoot && !!Array.isArray(expressionRoot)) {
          return [];
        }
        const suggestions = await (0, _helpers.suggestForExpression)({
          innerText,
          getColumnsByType: callbacks === null || callbacks === void 0 ? void 0 : callbacks.getByType,
          expressionRoot,
          location: _types.Location.STATS_WHERE,
          preferredExpressionType: 'boolean',
          context,
          hasMinimumLicenseRequired: callbacks === null || callbacks === void 0 ? void 0 : callbacks.hasMinimumLicenseRequired,
          activeProduct: context === null || context === void 0 ? void 0 : context.activeProduct
        });

        // Is this a complete boolean expression?
        // If so, we can call it done and suggest a pipe
        const expressionType = (0, _expressions.getExpressionType)(expressionRoot, context === null || context === void 0 ? void 0 : context.columns);
        if (expressionType === 'boolean' && (0, _expressions.isExpressionComplete)(expressionType, innerText)) {
          suggestions.push(_complete_items.pipeCompleteItem, {
            ..._complete_items.commaCompleteItem,
            text: ', '
          }, _complete_items.byCompleteItem);
        }
        return suggestions;
      }
    case 'grouping_expression_after_assignment':
      {
        var _context$histogramBar;
        // Find expression root
        const byNode = command.args[command.args.length - 1];
        const assignment = byNode.args[byNode.args.length - 1];
        const rightHandAssignment = (0, _is.isAssignment)(assignment) ? assignment.args[assignment.args.length - 1] : undefined;
        let expressionRoot = Array.isArray(rightHandAssignment) ? rightHandAssignment[0] : undefined;

        // @TODO the marker shouldn't be leaking through here
        if ((0, _ast.isMarkerNode)(expressionRoot)) {
          expressionRoot = undefined;
        }

        // guaranteed by the getPosition function, but we check it here for type safety
        if (Array.isArray(expressionRoot)) {
          return [];
        }
        const ignoredColumns = alreadyUsedColumns(command);
        return getExpressionSuggestions({
          innerText,
          expressionRoot,
          location: _types.Location.STATS_BY,
          context,
          callbacks,
          emptySuggestions: [(0, _complete_items.getDateHistogramCompletionItem)((_context$histogramBar = context === null || context === void 0 ? void 0 : context.histogramBarTarget) !== null && _context$histogramBar !== void 0 ? _context$histogramBar : 0)],
          afterCompleteSuggestions: (0, _utils.getCommaAndPipe)(innerText, expressionRoot, columnExists),
          advanceCursorAfterInitialColumn: false,
          ignoredColumns
        });
      }
    case 'grouping_expression_without_assignment':
      {
        var _callbacks$getSuggest2, _context$histogramBar2;
        let expressionRoot;
        if (!/,\s*$/.test(innerText)) {
          const byNode = command.args[command.args.length - 1];
          expressionRoot = byNode.args[byNode.args.length - 1];
        }
        // guaranteed by the getPosition function, but we check it here for type safety
        if (Array.isArray(expressionRoot)) {
          return [];
        }
        const ignoredColumns = alreadyUsedColumns(command);
        return getExpressionSuggestions({
          innerText,
          expressionRoot,
          location: _types.Location.STATS_BY,
          context,
          callbacks,
          emptySuggestions: [(0, _complete_items.getNewUserDefinedColumnSuggestion)((callbacks === null || callbacks === void 0 ? void 0 : (_callbacks$getSuggest2 = callbacks.getSuggestedUserDefinedColumnName) === null || _callbacks$getSuggest2 === void 0 ? void 0 : _callbacks$getSuggest2.call(callbacks)) || ''), (0, _complete_items.getDateHistogramCompletionItem)((_context$histogramBar2 = context === null || context === void 0 ? void 0 : context.histogramBarTarget) !== null && _context$histogramBar2 !== void 0 ? _context$histogramBar2 : 0)],
          afterCompleteSuggestions: (0, _utils.getCommaAndPipe)(innerText, expressionRoot, columnExists),
          advanceCursorAfterInitialColumn: false,
          ignoredColumns
        });
      }
    default:
      return [];
  }
}
async function getExpressionSuggestions({
  innerText,
  expressionRoot,
  location,
  context,
  callbacks,
  emptySuggestions = [],
  afterCompleteSuggestions = [],
  advanceCursorAfterInitialColumn,
  suggestColumns = true,
  ignoredColumns = []
}) {
  const suggestions = [];
  if (!(0, _utils.rightAfterColumn)(innerText, expressionRoot, name => (0, _helpers.columnExists)(name, context))) {
    suggestions.push(...(await (0, _helpers.suggestForExpression)({
      innerText,
      expressionRoot,
      location,
      hasMinimumLicenseRequired: callbacks === null || callbacks === void 0 ? void 0 : callbacks.hasMinimumLicenseRequired,
      activeProduct: context === null || context === void 0 ? void 0 : context.activeProduct,
      context,
      getColumnsByType: suggestColumns ? callbacks === null || callbacks === void 0 ? void 0 : callbacks.getByType : undefined,
      advanceCursorAfterInitialColumn,
      ignoredColumnsForEmptyExpression: ignoredColumns
    })));
  }
  if ((!expressionRoot || (0, _is.isColumn)(expressionRoot) && !(0, _helpers.columnExists)(expressionRoot.parts.join('.'), context)) && !/not\s+$/i.test(innerText)) {
    suggestions.push(...emptySuggestions);
  }
  if ((0, _expressions.isExpressionComplete)((0, _expressions.getExpressionType)(expressionRoot, context === null || context === void 0 ? void 0 : context.columns), innerText)) {
    suggestions.push(...afterCompleteSuggestions);
  }
  return suggestions;
}