"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.SynchronizationSubTaskRunner = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _server = require("@kbn/task-manager-plugin/server");
var _utils = require("../../utils");
var _constants = require("../../constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const LOOKBACK_WINDOW = 5 * 60 * 1000;
var ReindexStatus = /*#__PURE__*/function (ReindexStatus) {
  ReindexStatus["RUNNING"] = "running";
  ReindexStatus["COMPLETED"] = "completed";
  ReindexStatus["FAILED"] = "failed";
  ReindexStatus["MISSING_TASK_ID"] = "missing_task_id";
  return ReindexStatus;
}(ReindexStatus || {});
class SynchronizationSubTaskRunner {
  constructor({
    esReindexTaskId,
    lastSyncSuccess,
    lastSyncAttempt,
    sourceIndex,
    destIndex,
    owner,
    spaceId,
    syncType,
    esClient,
    logger
  }) {
    (0, _defineProperty2.default)(this, "sourceIndex", void 0);
    (0, _defineProperty2.default)(this, "destIndex", void 0);
    (0, _defineProperty2.default)(this, "owner", void 0);
    (0, _defineProperty2.default)(this, "spaceId", void 0);
    (0, _defineProperty2.default)(this, "syncType", void 0);
    (0, _defineProperty2.default)(this, "esClient", void 0);
    (0, _defineProperty2.default)(this, "logger", void 0);
    (0, _defineProperty2.default)(this, "errorSource", _server.TaskErrorSource.FRAMEWORK);
    (0, _defineProperty2.default)(this, "esReindexTaskId", void 0);
    (0, _defineProperty2.default)(this, "lastSyncSuccess", void 0);
    (0, _defineProperty2.default)(this, "lastSyncAttempt", void 0);
    if (lastSyncSuccess) this.lastSyncSuccess = new Date(lastSyncSuccess);
    if (lastSyncAttempt) this.lastSyncAttempt = new Date(lastSyncAttempt);
    this.esReindexTaskId = esReindexTaskId;
    this.sourceIndex = sourceIndex;
    this.destIndex = destIndex;
    this.owner = owner;
    this.spaceId = spaceId;
    this.syncType = syncType;
    this.esClient = esClient;
    this.logger = logger;
  }
  async run() {
    const esClient = this.esClient;
    try {
      // Parent task
      const destIndexExists = await esClient.indices.exists({
        index: this.destIndex
      });
      if (!destIndexExists) {
        this.logDebug('Destination index does not exist, skipping synchronization task.');
        return;
      }
      const previousReindexStatus = await this.getPreviousReindexStatus(esClient);
      this.logDebug(`Previous synchronization task status: "${previousReindexStatus}".`);
      // Break down previousReindexStatus into status for each sub task. Broken down by syncType.
      // </Parent task>

      // Sub task
      if (previousReindexStatus === ReindexStatus.RUNNING) {
        /*
         * If the reindex task is still running we return the
         * same state and the next run will cover any missing
         * updates.
         **/
        return this.getSyncState();
      }
      if (previousReindexStatus === ReindexStatus.COMPLETED) {
        /*
         * If the previous reindex task is completed we reindex
         * with a new time window.
         **/
        await this.isIndexAvailable(esClient);
        this.updateLastSyncTimes({
          updateSuccessTime: true
        });
        const esReindexTaskId = await this.synchronizeIndex({
          esClient
        });
        return {
          lastSyncSuccess: this.lastSyncSuccess,
          lastSyncAttempt: this.lastSyncAttempt,
          syncType: this.syncType,
          esReindexTaskId
        };
      }
      if (
      /*
       * A missing task id can only happen if this is
       * the first task execution.
       **/
      previousReindexStatus === ReindexStatus.MISSING_TASK_ID || previousReindexStatus === ReindexStatus.FAILED) {
        await this.isIndexAvailable(esClient);

        /*
         * There are two possible scenarios here:
         * 1. If the previous task failed (ReindexStatus.FAILED)
         * 2. If the state is missing
         *
         * In both cases we try to reindex without updating lastSyncSuccess.
         * This will ensure the reindex query is built with the correct value.
         **/
        this.updateLastSyncTimes({
          updateSuccessTime: false
        });
        const esReindexTaskId = await this.synchronizeIndex({
          esClient
        });
        return {
          lastSyncSuccess: this.lastSyncSuccess,
          lastSyncAttempt: this.lastSyncAttempt,
          syncType: this.syncType,
          esReindexTaskId
        };
      }
      throw new Error('Invalid task state.');
    } catch (e) {
      if ((0, _utils.isRetryableEsClientError)(e)) {
        (0, _server.throwRetryableError)((0, _server.createTaskRunError)(new Error(this.handleErrorMessage(e.message)), this.errorSource), true);
      }
      this.handleErrorMessage(e.message);
      (0, _server.throwUnrecoverableError)((0, _server.createTaskRunError)(e, this.errorSource));
    }
  }
  updateLastSyncTimes({
    updateSuccessTime
  }) {
    this.logDebug('Updating lastSyncAttempt and lastSyncSuccess before synchronization.');
    if (updateSuccessTime) {
      this.lastSyncSuccess = this.lastSyncAttempt;
    }
    this.lastSyncAttempt = new Date();
  }

  /**
   * This method does a call to elasticsearch that reindexes from this.destIndex
   * to this.sourceIndex. The query used takes into account the lastSyncSuccess
   * and lastSyncAttempt values in the class.
   *
   * @returns {SynchronizationTaskState} The updated task state
   */
  async synchronizeIndex({
    esClient
  }) {
    const painlessScript = await this.getPainlessScript(esClient);
    if (painlessScript.found) {
      this.logDebug(`Synchronizing with ${this.sourceIndex}.`);
      const reindexResponse = await esClient.reindex({
        source: {
          index: this.sourceIndex,
          query: this.buildSourceQuery()
        },
        dest: {
          index: this.destIndex
        },
        script: {
          id: painlessScript._id
        },
        /** If `true`, the request refreshes affected shards to make this operation visible to search. */
        refresh: true,
        /** We do not wait for the es reindex operation to be completed. */
        wait_for_completion: false
      });
      return reindexResponse.task;
    } else {
      throw (0, _server.createTaskRunError)(new Error(this.handleErrorMessage('Painless script not found.')), this.errorSource);
    }
  }
  async getPreviousReindexStatus(esClient) {
    var _taskResponse$respons, _taskResponse$respons2;
    this.logDebug('Checking previous synchronization task status.');
    if (!this.esReindexTaskId) {
      return ReindexStatus.MISSING_TASK_ID;
    }
    const taskResponse = await esClient.tasks.get({
      task_id: this.esReindexTaskId.toString()
    });
    if (!taskResponse.completed) {
      return ReindexStatus.RUNNING;
    }
    if ((_taskResponse$respons = taskResponse.response) !== null && _taskResponse$respons !== void 0 && (_taskResponse$respons2 = _taskResponse$respons.failures) !== null && _taskResponse$respons2 !== void 0 && _taskResponse$respons2.length || taskResponse !== null && taskResponse !== void 0 && taskResponse.error) {
      return ReindexStatus.FAILED;
    }
    return ReindexStatus.COMPLETED;
  }
  buildSourceQuery() {
    return _constants.SYNCHRONIZATION_QUERIES_DICTIONARY[this.syncType](new Date(this.lastSyncSuccess ? this.lastSyncSuccess : Date.now() - LOOKBACK_WINDOW), this.spaceId, this.owner);
  }
  getSyncState() {
    return {
      lastSyncSuccess: this.lastSyncSuccess,
      lastSyncAttempt: this.lastSyncAttempt,
      esReindexTaskId: this.esReindexTaskId,
      syncType: this.syncType
    };
  }
  async getMapping(esClient) {
    this.logDebug('Getting index mapping.');
    return esClient.indices.getMapping({
      index: this.destIndex
    });
  }
  async getPainlessScript(esClient) {
    const painlessScriptId = await this.getPainlessScriptId(esClient);
    this.logDebug(`Getting painless script with id: "${painlessScriptId}".`);
    return esClient.getScript({
      id: painlessScriptId
    });
  }
  async getPainlessScriptId(esClient) {
    var _mapping$this$destInd;
    const mapping = await this.getMapping(esClient);
    const painlessScriptId = (_mapping$this$destInd = mapping[this.destIndex].mappings._meta) === null || _mapping$this$destInd === void 0 ? void 0 : _mapping$this$destInd.painless_script_id;
    if (!painlessScriptId) {
      throw (0, _server.createTaskRunError)(new Error(this.handleErrorMessage('Painless script id missing from mapping.')), this.errorSource);
    }
    return painlessScriptId;
  }
  async isIndexAvailable(esClient) {
    this.logDebug('Checking index availability.');
    return esClient.cluster.health({
      index: this.destIndex,
      wait_for_status: 'green',
      timeout: '30s'
    });
  }
  logDebug(message) {
    this.logger.debug(`[${this.destIndex}] ${message}`, {
      tags: ['cai-synchronization', this.destIndex]
    });
  }
  handleErrorMessage(message) {
    const errorMessage = `[${this.destIndex}] Synchronization reindex failed. Error: ${message}`;
    this.logger.error(errorMessage, {
      tags: ['cai-synchronization', 'cai-synchronization-error', this.destIndex]
    });
    return errorMessage;
  }
  async cancel() {}
}
exports.SynchronizationSubTaskRunner = SynchronizationSubTaskRunner;