"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useResponsiveMenu = useResponsiveMenu;
var _react = require("react");
var _constants = require("../constants");
var _get_style_property = require("../utils/get_style_property");
var _use_raf_debounced = require("./use_raf_debounced");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * Custom hook for handling responsive menu behavior with dynamic height measurement
 * @param isCollapsed - Whether the side nav is collapsed
 * @param items - Navigation items
 * @returns Object with menu ref and partitioned menu items
 */
function useResponsiveMenu(isCollapsed, items) {
  const primaryMenuRef = (0, _react.useRef)(null);
  const heightsCacheRef = (0, _react.useRef)([]);
  const [visibleCount, setVisibleCount] = (0, _react.useState)(items.length);
  const visibleMenuItems = (0, _react.useMemo)(() => items.slice(0, visibleCount), [items, visibleCount]);
  const overflowMenuItems = (0, _react.useMemo)(() => items.slice(visibleCount), [items, visibleCount]);
  const stableItemsReference = useStableItemsReference(items);
  const recalculateMenuLayout = (0, _react.useCallback)(() => {
    if (!primaryMenuRef.current) return;

    // Primary menu
    const menu = primaryMenuRef.current;
    const menuHeight = menu.clientHeight;

    // 1. Cache the heights of all children
    cacheHeights(heightsCacheRef, menu, stableItemsReference);
    if (heightsCacheRef.current.length === 0) return;

    // Primary menu items
    const childrenHeights = heightsCacheRef.current;
    const childrenGap = (0, _get_style_property.getStyleProperty)(menu, 'gap');

    // 2. Calculate the number of visible menu items
    const nextVisibleCount = countVisibleItems(childrenHeights, childrenGap, menuHeight);

    // 3. Update the visible count if needed
    setVisibleCount(nextVisibleCount);
  }, [stableItemsReference]);
  const [scheduleRecalc, cancelRecalc] = (0, _use_raf_debounced.useRafDebouncedCallback)(recalculateMenuLayout);
  (0, _react.useLayoutEffect)(() => {
    // Invalidate the cache when items change
    setVisibleCount(stableItemsReference.length);
    heightsCacheRef.current = [];
    const observer = new ResizeObserver(() => {
      scheduleRecalc();
    });
    if (primaryMenuRef.current) {
      observer.observe(primaryMenuRef.current);
    }

    // Initial calculation
    scheduleRecalc();
    return () => {
      observer.disconnect();
      cancelRecalc();
    };
  }, [isCollapsed, stableItemsReference, scheduleRecalc, cancelRecalc]);
  return {
    primaryMenuRef,
    visibleMenuItems,
    overflowMenuItems
  };
}

/**
 * Utility function to cache the heights of the menu items in a ref.
 * It assumes one initial render where all items are in the DOM.
 *
 * @param ref - The ref to the heights cache.
 * @param menu - The menu element.
 * @param items - The menu items.
 */
const cacheHeights = (ref, menu, items) => {
  var _ref$current;
  if (((_ref$current = ref.current) === null || _ref$current === void 0 ? void 0 : _ref$current.length) !== items.length) {
    const children = Array.from(menu.children);

    // Only cache if the DOM has rendered all the items we expect
    if (children.length === items.length) {
      ref.current = children.map(child => child.clientHeight);
    }
  }
};

/**
 * Utility function to get the number of visible menu items until we reach the menu height or the limit of menu items.
 *
 * @param heights - The heights of the menu items.
 * @param gap - The gap between the menu items.
 * @param menuHeight - The height of the menu.
 *
 * @returns The number of visible menu items.
 */
const countVisibleItems = (heights, gap, menuHeight) => {
  const countItemsToFit = (availableHeight, limit) => {
    let itemCount = 0;
    let totalHeight = 0;
    for (let i = 0; i < heights.length && itemCount < limit; i++) {
      const itemHeight = heights[i];
      const nextTotalHeight = totalHeight + itemHeight + (itemCount > 0 ? gap : 0);
      if (nextTotalHeight <= availableHeight) {
        totalHeight = nextTotalHeight;
        itemCount++;
      } else {
        break;
      }
    }
    return itemCount;
  };

  // 1. Calculate how many items can fit without considering the "More" button
  const initialVisibleCount = countItemsToFit(menuHeight, _constants.MAX_MENU_ITEMS);

  // 2. If not all items are visible, we need the "More" button
  if (heights.length > initialVisibleCount) {
    const moreItemHeight = heights[0]; // Approximately the same height as any other item
    const availableHeight = menuHeight - moreItemHeight - gap;
    return countItemsToFit(availableHeight, _constants.MAX_MENU_ITEMS - 1);
  }
  return initialVisibleCount;
};

/**
 * Get a stable reference to the items array that changes only when we need to recalculate the height.
 * @param items - menu items
 */
const useStableItemsReference = items => {
  const ref = (0, _react.useRef)(items);
  const out = haveSameHeightSignature(ref.current, items) ? ref.current : items;

  // don’t write to a ref during render
  (0, _react.useLayoutEffect)(() => {
    if (!haveSameHeightSignature(ref.current, items)) {
      ref.current = items;
    }
  }, [items]);
  return out;
};

/**
 * Precheck to see if the item's height might have changed and if we need to do a full recalculation.
 * @param prev - prev menu items
 * @param next - next menu items
 */
const haveSameHeightSignature = (prev, next) => {
  if (prev === next) return true;
  if (prev.length !== next.length) return false;
  for (let i = 0; i < prev.length; i++) {
    // Only compare properties that might affect height
    if (prev[i].label !== next[i].label) return false;
  }
  return true;
};