"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getVulnerabilityTags = exports.getVulnerabilityRuleName = exports.generateVulnerabilitiesRuleQuery = exports.createDetectionRuleFromVulnerabilityFinding = void 0;
var _i18n = require("@kbn/i18n");
var _cloudSecurityPostureCommon = require("@kbn/cloud-security-posture-common");
var _is_native_csp_finding = require("@kbn/cloud-security-posture/src/utils/is_native_csp_finding");
var _get_vendor_name = require("@kbn/cloud-security-posture/src/utils/get_vendor_name");
var _constants = require("../../../../common/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const DEFAULT_RULE_RISK_SCORE = 0;
const DEFAULT_RULE_SEVERITY = 'low';
const DEFAULT_RULE_ENABLED = true;
const DEFAULT_RULE_AUTHOR = 'Elastic';
const DEFAULT_RULE_LICENSE = 'Elastic License v2';
const DEFAULT_MAX_ALERTS_PER_RULE = 100;
const ALERT_SUPPRESSION_FIELD = 'resource.id';
const ALERT_TIMESTAMP_FIELD = 'event.ingested';
const ALERT_SEVERITY_MAP_FIELD = 'vulnerability.severity';
const DEFAULT_INVESTIGATION_FIELDS = {
  field_names: ['resource.name', 'resource.id']
};
var RuleSeverityMapping = /*#__PURE__*/function (RuleSeverityMapping) {
  RuleSeverityMapping["Low"] = "low";
  RuleSeverityMapping["Medium"] = "medium";
  RuleSeverityMapping["High"] = "high";
  RuleSeverityMapping["Critical"] = "critical";
  return RuleSeverityMapping;
}(RuleSeverityMapping || {});
var AlertSuppressionMissingFieldsStrategy = /*#__PURE__*/function (AlertSuppressionMissingFieldsStrategy) {
  AlertSuppressionMissingFieldsStrategy["DoNotSuppress"] = "doNotSuppress";
  AlertSuppressionMissingFieldsStrategy["Suppress"] = "suppress";
  return AlertSuppressionMissingFieldsStrategy;
}(AlertSuppressionMissingFieldsStrategy || {});
const getVulnerabilityRuleName = vulnerability => {
  return _i18n.i18n.translate('xpack.csp.vulnerabilities.detectionRuleNamePrefix', {
    defaultMessage: 'Vulnerability: {vulnerabilityId}',
    values: {
      vulnerabilityId: Array.isArray(vulnerability.id) ? vulnerability.id[0] : vulnerability.id
    }
  });
};
exports.getVulnerabilityRuleName = getVulnerabilityRuleName;
const generateVulnerabilitiesRuleQuery = (vulnerability, startTimestamp = new Date().toISOString()) => {
  return `vulnerability.id: "${vulnerability.id}" AND event.ingested >= "${startTimestamp}"`;
};
exports.generateVulnerabilitiesRuleQuery = generateVulnerabilitiesRuleQuery;
const CSP_RULE_TAG = 'Cloud Security';
const CNVM_TAG = 'CNVM';
const CNVM_RULE_TAG_DATA_SOURCE = 'Data Source: Cloud Native Vulnerability Management';
const CNVM_RULE_TAG_USE_CASE = 'Use Case: Vulnerability';
const CNVM_RULE_TAG_OS = 'OS: Linux';
const getVulnerabilityTags = vulnerabilityFinding => {
  var _vulnerabilityFinding, _vulnerabilityFinding2, _vulnerabilityFinding3;
  const {
    id
  } = (_vulnerabilityFinding = vulnerabilityFinding.vulnerability) !== null && _vulnerabilityFinding !== void 0 ? _vulnerabilityFinding : {};
  // we used spread operator to make sure we are not mutating the original array
  let tags = Array.isArray(id) ? [...id] : [id];
  const vendor = ((_vulnerabilityFinding2 = vulnerabilityFinding.observer) === null || _vulnerabilityFinding2 === void 0 ? void 0 : _vulnerabilityFinding2.vendor) || (vulnerabilityFinding === null || vulnerabilityFinding === void 0 ? void 0 : (_vulnerabilityFinding3 = vulnerabilityFinding.data_stream) === null || _vulnerabilityFinding3 === void 0 ? void 0 : _vulnerabilityFinding3.dataset);
  if ((0, _is_native_csp_finding.isNativeCspFinding)(vulnerabilityFinding)) {
    tags = [CNVM_TAG, CNVM_RULE_TAG_DATA_SOURCE, CNVM_RULE_TAG_USE_CASE, CNVM_RULE_TAG_OS, ...tags];
  } else if (!!vendor) {
    tags.push(vendor);
  }
  return [CSP_RULE_TAG, ...tags];
};

/*
 * Creates a detection rule from a Vulnerability
 */
exports.getVulnerabilityTags = getVulnerabilityTags;
const createDetectionRuleFromVulnerabilityFinding = async (http, vulnerabilityFinding) => {
  var _vulnerabilityFinding4;
  if (((_vulnerabilityFinding4 = vulnerabilityFinding.data_stream) === null || _vulnerabilityFinding4 === void 0 ? void 0 : _vulnerabilityFinding4.dataset) !== _get_vendor_name.CSP_VULN_DATASET) {
    throw new Error(_i18n.i18n.translate('xpack.csp.createDetectionRuleFromVulnerabilityFinding.createRuleErrorMessage', {
      defaultMessage: 'Rule creation is currently only available for Elastic findings'
    }));
  }
  if (!vulnerabilityFinding.vulnerability.id) {
    throw new Error(_i18n.i18n.translate('xpack.csp.createDetectionRuleFromVulnerabilityFinding.createRuleErrorMessageMissingId', {
      defaultMessage: 'The finding document is missing vulnerability.id field'
    }));
  }
  const tags = getVulnerabilityTags(vulnerabilityFinding);
  const vulnerability = vulnerabilityFinding.vulnerability;
  const getReferences = () => {
    if (!vulnerability.reference) return [];
    return Array.isArray(vulnerability.reference) ? vulnerability.reference : [vulnerability.reference];
  };
  return await (0, _cloudSecurityPostureCommon.createDetectionRule)({
    http,
    rule: {
      type: 'query',
      language: 'kuery',
      license: DEFAULT_RULE_LICENSE,
      author: [DEFAULT_RULE_AUTHOR],
      filters: [],
      false_positives: [],
      risk_score: DEFAULT_RULE_RISK_SCORE,
      risk_score_mapping: [],
      severity: DEFAULT_RULE_SEVERITY,
      severity_mapping: [{
        field: ALERT_SEVERITY_MAP_FIELD,
        value: _cloudSecurityPostureCommon.VULNERABILITIES_SEVERITY.LOW,
        operator: 'equals',
        severity: RuleSeverityMapping.Low
      }, {
        field: ALERT_SEVERITY_MAP_FIELD,
        value: _cloudSecurityPostureCommon.VULNERABILITIES_SEVERITY.MEDIUM,
        operator: 'equals',
        severity: RuleSeverityMapping.Medium
      }, {
        field: ALERT_SEVERITY_MAP_FIELD,
        value: _cloudSecurityPostureCommon.VULNERABILITIES_SEVERITY.HIGH,
        operator: 'equals',
        severity: RuleSeverityMapping.High
      }, {
        field: ALERT_SEVERITY_MAP_FIELD,
        value: _cloudSecurityPostureCommon.VULNERABILITIES_SEVERITY.CRITICAL,
        operator: 'equals',
        severity: RuleSeverityMapping.Critical
      }, {
        field: ALERT_SEVERITY_MAP_FIELD,
        value: _cloudSecurityPostureCommon.VULNERABILITIES_SEVERITY.UNKNOWN,
        operator: 'equals',
        severity: RuleSeverityMapping.Low
      }],
      threat: [],
      interval: '1h',
      from: `now-${_cloudSecurityPostureCommon.LATEST_VULNERABILITIES_RETENTION_POLICY}`,
      to: 'now',
      max_signals: DEFAULT_MAX_ALERTS_PER_RULE,
      timestamp_override: ALERT_TIMESTAMP_FIELD,
      timestamp_override_fallback_disabled: false,
      actions: [],
      enabled: DEFAULT_RULE_ENABLED,
      alert_suppression: {
        group_by: [ALERT_SUPPRESSION_FIELD],
        missing_fields_strategy: AlertSuppressionMissingFieldsStrategy.Suppress
      },
      index: [_constants.VULNERABILITIES_INDEX_PATTERN],
      query: generateVulnerabilitiesRuleQuery(vulnerability),
      references: getReferences(),
      name: getVulnerabilityRuleName(vulnerability),
      description: vulnerability.description,
      tags,
      investigation_fields: DEFAULT_INVESTIGATION_FIELDS
    }
  });
};
exports.createDetectionRuleFromVulnerabilityFinding = createDetectionRuleFromVulnerabilityFinding;