"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.saveDiscoverSession = void 0;
var _uuid = require("uuid");
var _esQuery = require("@kbn/es-query");
var _i18n = require("@kbn/i18n");
var _lodash = require("lodash");
var _dataViewUtils = require("@kbn/data-view-utils");
var _selectors = require("../selectors");
var _utils = require("../utils");
var _runtime_state = require("../runtime_state");
var _tab_mapping_utils = require("../tab_mapping_utils");
var _data_views = require("./data_views");
var _reset_discover_session = require("./reset_discover_session");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const saveDiscoverSession = exports.saveDiscoverSession = (0, _utils.createInternalStateAsyncThunk)('internalState/saveDiscoverSession', async ({
  newTitle,
  newCopyOnSave,
  newTimeRestore,
  newDescription,
  newTags,
  isTitleDuplicateConfirmed,
  onTitleDuplicate
}, {
  dispatch,
  getState,
  extra: {
    services,
    runtimeStateManager
  }
}) => {
  var _state$persistedDisco, _state$persistedDisco2;
  const state = getState();
  const currentTabs = (0, _selectors.selectAllTabs)(state);
  const adHocDataViews = new Map();
  let nextSelectedTabId = state.tabs.unsafeCurrentId;
  const selectedTab = currentTabs.find(tab => tab.id === state.tabs.unsafeCurrentId);
  const updatedTabs = await Promise.all(currentTabs.map(async tab => {
    const tabRuntimeState = (0, _runtime_state.selectTabRuntimeState)(runtimeStateManager, tab.id);
    const tabStateContainer = tabRuntimeState.stateContainer$.getValue();
    const overriddenVisContextAfterInvalidation = tab.overriddenVisContextAfterInvalidation;
    let updatedTab;
    if (tabStateContainer) {
      updatedTab = (0, _lodash.cloneDeep)({
        ...(0, _tab_mapping_utils.fromSavedSearchToSavedObjectTab)({
          tab,
          savedSearch: tabStateContainer.savedSearchState.getState(),
          services
        }),
        timeRestore: newTimeRestore,
        timeRange: newTimeRestore ? tab.globalState.timeRange : undefined,
        refreshInterval: newTimeRestore ? tab.globalState.refreshInterval : undefined
      });
    } else {
      updatedTab = (0, _lodash.cloneDeep)((0, _tab_mapping_utils.fromTabStateToSavedObjectTab)({
        tab,
        timeRestore: newTimeRestore,
        services
      }));
      if (newTimeRestore && !updatedTab.timeRange && selectedTab !== null && selectedTab !== void 0 && selectedTab.globalState.timeRange) {
        // assign the current time range of the selected tab if time restore is enabled and no time range was set yet for this tab
        updatedTab.timeRange = selectedTab.globalState.timeRange;
        updatedTab.refreshInterval = selectedTab.globalState.refreshInterval;
      }
    }
    if (newCopyOnSave) {
      // to avoid id conflicts, we need to assign a new id to the tab if we're copying a discover session
      const newTabId = (0, _uuid.v4)();
      if (tab.id === nextSelectedTabId) {
        nextSelectedTabId = newTabId;
      }
      updatedTab.id = newTabId;
    }
    if (overriddenVisContextAfterInvalidation) {
      updatedTab.visContext = overriddenVisContextAfterInvalidation;
    }
    const dataViewSpec = updatedTab.serializedSearchSource.index;

    // If the data view is a non-ES|QL ad hoc data view, it may need to be cloned
    if ((0, _lodash.isObject)(dataViewSpec) && dataViewSpec.id && dataViewSpec.type !== _dataViewUtils.ESQL_TYPE) {
      let action;
      if (state.defaultProfileAdHocDataViewIds.includes(dataViewSpec.id)) {
        // If the Discover session is using a default profile ad hoc data view,
        // we copy it with a new ID to avoid conflicts with the profile defaults
        action = 'copy';
      } else if (newCopyOnSave) {
        // Otherwise, if we're copying a session with a custom ad hoc data view,
        // we replace it with a cloned one to avoid ID conflicts across sessions
        action = 'replace';
      }
      if (action) {
        var _adHocDataViews$get;
        const adHocEntry = (_adHocDataViews$get = adHocDataViews.get(dataViewSpec.id)) !== null && _adHocDataViews$get !== void 0 ? _adHocDataViews$get : {
          dataViewSpec,
          action,
          tabs: []
        };
        adHocEntry.tabs.push(updatedTab);
        adHocDataViews.set(dataViewSpec.id, adHocEntry);
      }
    }
    return updatedTab;
  }));
  for (const adHocEntry of adHocDataViews.values()) {
    const {
      dataViewSpec,
      action,
      tabs
    } = adHocEntry;
    if (!dataViewSpec.id) {
      continue;
    }
    let newDataViewSpec;
    if (action === 'copy') {
      var _dataViewSpec$name;
      newDataViewSpec = {
        ...dataViewSpec,
        id: (0, _uuid.v4)(),
        name: _i18n.i18n.translate('discover.savedSearch.defaultProfileDataViewCopyName', {
          defaultMessage: '{dataViewName} ({discoverSessionTitle})',
          values: {
            dataViewName: (_dataViewSpec$name = dataViewSpec.name) !== null && _dataViewSpec$name !== void 0 ? _dataViewSpec$name : dataViewSpec.title,
            discoverSessionTitle: newTitle
          }
        }),
        managed: false
      };
      const dataView = await services.dataViews.create(newDataViewSpec);

      // Make sure our state is aware of the copy so it appears in the UI
      dispatch((0, _data_views.appendAdHocDataViews)(dataView));
    } else {
      newDataViewSpec = {
        ...dataViewSpec,
        id: (0, _uuid.v4)(),
        managed: false
      };

      // Clear out the old data view since it's no longer needed
      services.dataViews.clearInstanceCache(dataViewSpec.id);
      const dataView = await services.dataViews.create(newDataViewSpec);

      // Make sure our state is aware of the new data view
      dispatch((0, _data_views.replaceAdHocDataViewWithId)(dataViewSpec.id, dataView));
    }

    // Update all applicable tabs to use the new data view spec
    for (const tab of tabs) {
      tab.serializedSearchSource.index = newDataViewSpec;

      // We also need to update the filter references
      if (Array.isArray(tab.serializedSearchSource.filter)) {
        tab.serializedSearchSource.filter = (0, _esQuery.updateFilterReferences)(tab.serializedSearchSource.filter, dataViewSpec.id, newDataViewSpec.id);
      }
    }
  }
  const saveParams = {
    id: (_state$persistedDisco = state.persistedDiscoverSession) === null || _state$persistedDisco === void 0 ? void 0 : _state$persistedDisco.id,
    title: newTitle,
    description: newDescription,
    tabs: updatedTabs,
    tags: services.savedObjectsTagging ? newTags : (_state$persistedDisco2 = state.persistedDiscoverSession) === null || _state$persistedDisco2 === void 0 ? void 0 : _state$persistedDisco2.tags
  };
  const saveOptions = {
    onTitleDuplicate,
    copyOnSave: newCopyOnSave,
    isTitleDuplicateConfirmed
  };
  const discoverSession = await services.savedSearch.saveDiscoverSession(saveParams, saveOptions);
  if (discoverSession) {
    await dispatch((0, _reset_discover_session.resetDiscoverSession)({
      updatedDiscoverSession: discoverSession,
      nextSelectedTabId
    })).unwrap();
  }
  return {
    discoverSession,
    nextSelectedTabId
  };
});