/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.monitor.os;

import java.io.IOException;
import java.lang.management.ManagementFactory;
import java.lang.management.OperatingSystemMXBean;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.math.BigInteger;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Stream;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.lucene.util.Constants;
import org.elasticsearch.common.unit.Processors;
import org.elasticsearch.core.PathUtils;
import org.elasticsearch.core.SuppressForbidden;
import org.elasticsearch.monitor.Probes;
import org.elasticsearch.monitor.os.OsInfo;
import org.elasticsearch.monitor.os.OsStats;

public class OsProbe {
    private static final OperatingSystemMXBean osMxBean = ManagementFactory.getOperatingSystemMXBean();
    private static final String memoryOverrideProperty = System.getProperty("es.total_memory_bytes");
    private static final Method getFreePhysicalMemorySize = OsProbe.getMethod("getFreePhysicalMemorySize");
    private static final Method getTotalPhysicalMemorySize = OsProbe.getMethod("getTotalPhysicalMemorySize");
    private static final Method getFreeSwapSpaceSize = OsProbe.getMethod("getFreeSwapSpaceSize");
    private static final Method getTotalSwapSpaceSize = OsProbe.getMethod("getTotalSwapSpaceSize");
    private static final Method getSystemLoadAverage = OsProbe.getMethod("getSystemLoadAverage");
    private static final Method getSystemCpuLoad = OsProbe.getMethod("getSystemCpuLoad");
    private static final String CONTROL_GROUPS_HIERARCHY_OVERRIDE = System.getProperty("es.cgroups.hierarchy.override");
    private static final Logger logger = LogManager.getLogger(OsProbe.class);

    public long getFreePhysicalMemorySize() {
        if (getFreePhysicalMemorySize == null) {
            logger.warn("getFreePhysicalMemorySize is not available");
            return 0L;
        }
        try {
            long freeMem = (Long)getFreePhysicalMemorySize.invoke((Object)osMxBean, new Object[0]);
            if (freeMem < 0L) {
                logger.debug("OS reported a negative free memory value [{}]", (Object)freeMem);
                return 0L;
            }
            return freeMem;
        }
        catch (Exception e) {
            logger.warn("exception retrieving free physical memory", (Throwable)e);
            return 0L;
        }
    }

    public long getTotalPhysicalMemorySize() {
        if (getTotalPhysicalMemorySize == null) {
            logger.warn("getTotalPhysicalMemorySize is not available");
            return 0L;
        }
        try {
            long totalMem = (Long)getTotalPhysicalMemorySize.invoke((Object)osMxBean, new Object[0]);
            if (totalMem < 0L) {
                logger.debug("OS reported a negative total memory value [{}]", (Object)totalMem);
                return 0L;
            }
            if (totalMem == 0L && this.isDebian8()) {
                totalMem = this.getTotalMemFromProcMeminfo();
            }
            return totalMem;
        }
        catch (Exception e) {
            logger.warn("exception retrieving total physical memory", (Throwable)e);
            return 0L;
        }
    }

    public long getAdjustedTotalMemorySize() {
        return Optional.ofNullable(OsProbe.getTotalMemoryOverride(memoryOverrideProperty)).orElse(this.getTotalPhysicalMemorySize());
    }

    static Long getTotalMemoryOverride(String memoryOverrideProperty) {
        if (memoryOverrideProperty == null) {
            return null;
        }
        try {
            long memoryOverride = Long.parseLong(memoryOverrideProperty);
            if (memoryOverride < 0L) {
                throw new IllegalArgumentException("Negative memory size specified in [es.total_memory_bytes]: [" + memoryOverrideProperty + "]");
            }
            return memoryOverride;
        }
        catch (NumberFormatException e) {
            throw new IllegalArgumentException("Invalid value for [es.total_memory_bytes]: [" + memoryOverrideProperty + "]", e);
        }
    }

    public long getFreeSwapSpaceSize() {
        if (getFreeSwapSpaceSize == null) {
            logger.warn("getFreeSwapSpaceSize is not available");
            return 0L;
        }
        try {
            long mem = (Long)getFreeSwapSpaceSize.invoke((Object)osMxBean, new Object[0]);
            if (mem < 0L) {
                logger.debug("OS reported a negative free swap space size [{}]", (Object)mem);
                return 0L;
            }
            return mem;
        }
        catch (Exception e) {
            logger.warn("exception retrieving free swap space size", (Throwable)e);
            return 0L;
        }
    }

    public long getTotalSwapSpaceSize() {
        if (getTotalSwapSpaceSize == null) {
            logger.warn("getTotalSwapSpaceSize is not available");
            return 0L;
        }
        try {
            long mem = (Long)getTotalSwapSpaceSize.invoke((Object)osMxBean, new Object[0]);
            if (mem < 0L) {
                logger.debug("OS reported a negative total swap space size [{}]", (Object)mem);
                return 0L;
            }
            return mem;
        }
        catch (Exception e) {
            logger.warn("exception retrieving total swap space size", (Throwable)e);
            return 0L;
        }
    }

    final double[] getSystemLoadAverage() {
        if (Constants.WINDOWS) {
            return null;
        }
        if (Constants.LINUX) {
            try {
                String procLoadAvg = this.readProcLoadavg();
                assert (procLoadAvg.matches("(\\d+\\.\\d+\\s+){3}\\d+/\\d+\\s+\\d+"));
                String[] fields = procLoadAvg.split("\\s+");
                return new double[]{Double.parseDouble(fields[0]), Double.parseDouble(fields[1]), Double.parseDouble(fields[2])};
            }
            catch (IOException e) {
                if (logger.isDebugEnabled()) {
                    logger.debug("error reading /proc/loadavg", (Throwable)e);
                }
                return null;
            }
        }
        assert (Constants.MAC_OS_X);
        if (getSystemLoadAverage == null) {
            return null;
        }
        try {
            double oneMinuteLoadAverage = (Double)getSystemLoadAverage.invoke((Object)osMxBean, new Object[0]);
            return new double[]{oneMinuteLoadAverage >= 0.0 ? oneMinuteLoadAverage : -1.0, -1.0, -1.0};
        }
        catch (IllegalAccessException | InvocationTargetException e) {
            if (logger.isDebugEnabled()) {
                logger.debug("error reading one minute load average from operating system", (Throwable)e);
            }
            return null;
        }
    }

    @SuppressForbidden(reason="access /proc/loadavg")
    String readProcLoadavg() throws IOException {
        return OsProbe.readSingleLine(PathUtils.get((String)"/proc/loadavg", (String[])new String[0]));
    }

    public static short getSystemCpuPercent() {
        return Probes.getLoadAndScaleToPercent(getSystemCpuLoad, osMxBean);
    }

    private static String readSingleLine(Path path) throws IOException {
        List<String> lines = Files.readAllLines(path);
        assert (lines.size() == 1) : String.join((CharSequence)"\n", lines);
        return lines.get(0);
    }

    private Map<String, String> getControlGroups() throws IOException {
        List<String> lines = this.readProcSelfCgroup();
        HashMap<String, String> controllerMap = new HashMap<String, String>();
        for (String line : lines) {
            String[] controllers;
            String[] fields = line.split(":");
            assert (fields.length == 3);
            for (String controller : controllers = fields[1].split(",")) {
                String controlGroupPath = CONTROL_GROUPS_HIERARCHY_OVERRIDE != null ? CONTROL_GROUPS_HIERARCHY_OVERRIDE : fields[2];
                String previous = controllerMap.put(controller, controlGroupPath);
                assert (previous == null);
            }
        }
        return controllerMap;
    }

    @SuppressForbidden(reason="access /proc/self/cgroup")
    List<String> readProcSelfCgroup() throws IOException {
        List<String> lines = Files.readAllLines(PathUtils.get((String)"/proc/self/cgroup", (String[])new String[0]));
        assert (lines != null && !lines.isEmpty());
        return lines;
    }

    private BigInteger getCgroupCpuAcctUsageNanos(String controlGroup) throws IOException {
        return new BigInteger(this.readSysFsCgroupCpuAcctCpuAcctUsage(controlGroup));
    }

    @SuppressForbidden(reason="access /sys/fs/cgroup/cpuacct")
    String readSysFsCgroupCpuAcctCpuAcctUsage(String controlGroup) throws IOException {
        return OsProbe.readSingleLine(PathUtils.get((String)"/sys/fs/cgroup/cpuacct", (String[])new String[]{controlGroup, "cpuacct.usage"}));
    }

    private long[] getCgroupV2CpuLimit(String controlGroup) throws IOException {
        String entry = this.readCgroupV2CpuLimit(controlGroup);
        String[] parts = entry.split("\\s+");
        assert (parts.length == 2) : "Expected 2 fields in [cpu.max]";
        long[] values = new long[]{"max".equals(parts[0]) ? -1L : Long.parseLong(parts[0]), Long.parseLong(parts[1])};
        return values;
    }

    @SuppressForbidden(reason="access /sys/fs/cgroup/cpu.max")
    String readCgroupV2CpuLimit(String controlGroup) throws IOException {
        return OsProbe.readSingleLine(PathUtils.get((String)"/sys/fs/cgroup/", (String[])new String[]{controlGroup, "cpu.max"}));
    }

    private long getCgroupCpuAcctCpuCfsPeriodMicros(String controlGroup) throws IOException {
        return Long.parseLong(this.readSysFsCgroupCpuAcctCpuCfsPeriod(controlGroup));
    }

    @SuppressForbidden(reason="access /sys/fs/cgroup/cpu")
    String readSysFsCgroupCpuAcctCpuCfsPeriod(String controlGroup) throws IOException {
        return OsProbe.readSingleLine(PathUtils.get((String)"/sys/fs/cgroup/cpu", (String[])new String[]{controlGroup, "cpu.cfs_period_us"}));
    }

    private long getCgroupCpuAcctCpuCfsQuotaMicros(String controlGroup) throws IOException {
        return Long.parseLong(this.readSysFsCgroupCpuAcctCpuAcctCfsQuota(controlGroup));
    }

    @SuppressForbidden(reason="access /sys/fs/cgroup/cpu")
    String readSysFsCgroupCpuAcctCpuAcctCfsQuota(String controlGroup) throws IOException {
        return OsProbe.readSingleLine(PathUtils.get((String)"/sys/fs/cgroup/cpu", (String[])new String[]{controlGroup, "cpu.cfs_quota_us"}));
    }

    private OsStats.Cgroup.CpuStat getCgroupCpuAcctCpuStat(String controlGroup) throws IOException {
        BigInteger SENTINEL_VALUE = BigInteger.valueOf(-1L);
        List<String> lines = this.readSysFsCgroupCpuAcctCpuStat(controlGroup);
        BigInteger numberOfPeriods = SENTINEL_VALUE;
        BigInteger numberOfTimesThrottled = SENTINEL_VALUE;
        BigInteger timeThrottledNanos = SENTINEL_VALUE;
        for (String line : lines) {
            String[] fields = line.split("\\s+");
            switch (fields[0]) {
                case "nr_periods": {
                    numberOfPeriods = new BigInteger(fields[1]);
                    break;
                }
                case "nr_throttled": {
                    numberOfTimesThrottled = new BigInteger(fields[1]);
                    break;
                }
                case "throttled_time": {
                    timeThrottledNanos = new BigInteger(fields[1]);
                }
            }
        }
        assert (!numberOfPeriods.equals(SENTINEL_VALUE));
        assert (!numberOfTimesThrottled.equals(SENTINEL_VALUE));
        assert (!timeThrottledNanos.equals(SENTINEL_VALUE));
        return new OsStats.Cgroup.CpuStat(numberOfPeriods, numberOfTimesThrottled, timeThrottledNanos);
    }

    @SuppressForbidden(reason="access /sys/fs/cgroup/cpu")
    List<String> readSysFsCgroupCpuAcctCpuStat(String controlGroup) throws IOException {
        List<String> lines = Files.readAllLines(PathUtils.get((String)"/sys/fs/cgroup/cpu", (String[])new String[]{controlGroup, "cpu.stat"}));
        assert (lines != null && lines.size() >= 3);
        return lines;
    }

    private String getCgroupMemoryLimitInBytes(String controlGroup) throws IOException {
        return this.readSysFsCgroupMemoryLimitInBytes(controlGroup);
    }

    @SuppressForbidden(reason="access /sys/fs/cgroup/memory")
    String readSysFsCgroupMemoryLimitInBytes(String controlGroup) throws IOException {
        return OsProbe.readSingleLine(PathUtils.get((String)"/sys/fs/cgroup/memory", (String[])new String[]{controlGroup, "memory.limit_in_bytes"}));
    }

    private String getCgroupV2MemoryLimitInBytes(String controlGroup) throws IOException {
        return this.readSysFsCgroupV2MemoryLimitInBytes(controlGroup);
    }

    @SuppressForbidden(reason="access /sys/fs/cgroup/memory.max")
    String readSysFsCgroupV2MemoryLimitInBytes(String controlGroup) throws IOException {
        return OsProbe.readSingleLine(PathUtils.get((String)"/sys/fs/cgroup/", (String[])new String[]{controlGroup, "memory.max"}));
    }

    private String getCgroupMemoryUsageInBytes(String controlGroup) throws IOException {
        return this.readSysFsCgroupMemoryUsageInBytes(controlGroup);
    }

    @SuppressForbidden(reason="access /sys/fs/cgroup/memory")
    String readSysFsCgroupMemoryUsageInBytes(String controlGroup) throws IOException {
        return OsProbe.readSingleLine(PathUtils.get((String)"/sys/fs/cgroup/memory", (String[])new String[]{controlGroup, "memory.usage_in_bytes"}));
    }

    private String getCgroupV2MemoryUsageInBytes(String controlGroup) throws IOException {
        return this.readSysFsCgroupV2MemoryUsageInBytes(controlGroup);
    }

    @SuppressForbidden(reason="access /sys/fs/cgroup/memory.current")
    String readSysFsCgroupV2MemoryUsageInBytes(String controlGroup) throws IOException {
        return OsProbe.readSingleLine(PathUtils.get((String)"/sys/fs/cgroup/", (String[])new String[]{controlGroup, "memory.current"}));
    }

    @SuppressForbidden(reason="access /proc/self/cgroup, /sys/fs/cgroup/cpu, /sys/fs/cgroup/cpuacct and /sys/fs/cgroup/memory")
    boolean areCgroupStatsAvailable() throws IOException {
        if (!Files.exists(PathUtils.get((String)"/proc/self/cgroup", (String[])new String[0]), new LinkOption[0])) {
            return false;
        }
        List<String> lines = this.readProcSelfCgroup();
        if (lines.size() == 1 && lines.get(0).startsWith("0::")) {
            return Stream.of("/sys/fs/cgroup/cpu.stat", "/sys/fs/cgroup/memory.stat").allMatch(path -> Files.exists(PathUtils.get((String)path, (String[])new String[0]), new LinkOption[0]));
        }
        return Stream.of("/sys/fs/cgroup/cpu", "/sys/fs/cgroup/cpuacct", "/sys/fs/cgroup/memory").allMatch(path -> Files.exists(PathUtils.get((String)path, (String[])new String[0]), new LinkOption[0]));
    }

    @SuppressForbidden(reason="Uses PathUtils.get to generate meaningful assertion messages")
    private Map<String, BigInteger> getCgroupV2CpuStats(String controlGroup) throws IOException {
        List<String> lines = this.readCgroupV2CpuStats(controlGroup);
        HashMap<String, BigInteger> stats = new HashMap<String, BigInteger>();
        BigInteger SENTINEL_VALUE = BigInteger.valueOf(-1L);
        for (String line : lines) {
            String[] parts = line.split("\\s+");
            assert (parts.length == 2) : "Corrupt cpu.stat line: [" + line + "]";
            stats.put(parts[0], new BigInteger(parts[1]));
        }
        List<String> expectedKeys = List.of("system_usec", "usage_usec", "user_usec");
        expectedKeys.forEach(key -> {
            assert (stats.containsKey(key)) : "[" + key + "] missing from " + String.valueOf(PathUtils.get((String)"/sys/fs/cgroup", (String[])new String[]{controlGroup, "cpu.stat"}));
            assert (((BigInteger)stats.get(key)).compareTo(SENTINEL_VALUE) != 0) : ((BigInteger)stats.get(key)).toString();
        });
        List<String> optionalKeys = List.of("nr_periods", "nr_throttled", "throttled_usec");
        optionalKeys.forEach(key -> {
            if (!stats.containsKey(key)) {
                stats.put((String)key, BigInteger.ZERO);
            }
            assert (((BigInteger)stats.get(key)).compareTo(SENTINEL_VALUE) != 0) : "[" + key + "] in " + String.valueOf(PathUtils.get((String)"/sys/fs/cgroup", (String[])new String[]{controlGroup, "cpu.stat"})) + " is -1";
        });
        return stats;
    }

    @SuppressForbidden(reason="access /sys/fs/cgroup/cpu.stat")
    List<String> readCgroupV2CpuStats(String controlGroup) throws IOException {
        return Files.readAllLines(PathUtils.get((String)"/sys/fs/cgroup", (String[])new String[]{controlGroup, "cpu.stat"}));
    }

    private OsStats.Cgroup getCgroup() {
        try {
            String cgroupMemoryUsageInBytes;
            String cgroupMemoryLimitInBytes;
            OsStats.Cgroup.CpuStat cpuStat;
            long cgroupCpuAcctCpuCfsPeriodMicros;
            long cgroupCpuAcctCpuCfsQuotaMicros;
            BigInteger cgroupCpuAcctUsageNanos;
            String cpuControlGroup;
            String cpuAcctControlGroup;
            String memoryControlGroup;
            if (!this.areCgroupStatsAvailable()) {
                return null;
            }
            Map<String, String> controllerMap = this.getControlGroups();
            assert (!controllerMap.isEmpty());
            if (controllerMap.size() == 1 && controllerMap.containsKey("")) {
                cpuAcctControlGroup = memoryControlGroup = controllerMap.get("");
                cpuControlGroup = memoryControlGroup;
                Map<String, BigInteger> cpuStatsMap = this.getCgroupV2CpuStats(cpuControlGroup);
                BigInteger THOUSAND = BigInteger.valueOf(1000L);
                cgroupCpuAcctUsageNanos = cpuStatsMap.get("usage_usec").multiply(THOUSAND);
                long[] cpuLimits = this.getCgroupV2CpuLimit(cpuControlGroup);
                cgroupCpuAcctCpuCfsQuotaMicros = cpuLimits[0];
                cgroupCpuAcctCpuCfsPeriodMicros = cpuLimits[1];
                cpuStat = new OsStats.Cgroup.CpuStat(cpuStatsMap.get("nr_periods"), cpuStatsMap.get("nr_throttled"), cpuStatsMap.get("throttled_usec").multiply(THOUSAND));
                cgroupMemoryLimitInBytes = this.getCgroupV2MemoryLimitInBytes(memoryControlGroup);
                cgroupMemoryUsageInBytes = this.getCgroupV2MemoryUsageInBytes(memoryControlGroup);
            } else {
                cpuAcctControlGroup = controllerMap.get("cpuacct");
                if (cpuAcctControlGroup == null) {
                    logger.debug("no [cpuacct] data found in cgroup stats");
                    return null;
                }
                cgroupCpuAcctUsageNanos = this.getCgroupCpuAcctUsageNanos(cpuAcctControlGroup);
                cpuControlGroup = controllerMap.get("cpu");
                if (cpuControlGroup == null) {
                    logger.debug("no [cpu] data found in cgroup stats");
                    return null;
                }
                cgroupCpuAcctCpuCfsPeriodMicros = this.getCgroupCpuAcctCpuCfsPeriodMicros(cpuControlGroup);
                cgroupCpuAcctCpuCfsQuotaMicros = this.getCgroupCpuAcctCpuCfsQuotaMicros(cpuControlGroup);
                cpuStat = this.getCgroupCpuAcctCpuStat(cpuControlGroup);
                memoryControlGroup = controllerMap.get("memory");
                if (memoryControlGroup == null) {
                    logger.debug("no [memory] data found in cgroup stats");
                    return null;
                }
                cgroupMemoryLimitInBytes = this.getCgroupMemoryLimitInBytes(memoryControlGroup);
                cgroupMemoryUsageInBytes = this.getCgroupMemoryUsageInBytes(memoryControlGroup);
            }
            return new OsStats.Cgroup(cpuAcctControlGroup, cgroupCpuAcctUsageNanos, cpuControlGroup, cgroupCpuAcctCpuCfsPeriodMicros, cgroupCpuAcctCpuCfsQuotaMicros, cpuStat, memoryControlGroup, cgroupMemoryLimitInBytes, cgroupMemoryUsageInBytes);
        }
        catch (IOException e) {
            logger.debug("error reading control group stats", (Throwable)e);
            return null;
        }
    }

    public static OsProbe getInstance() {
        return OsProbeHolder.INSTANCE;
    }

    OsProbe() {
    }

    OsInfo osInfo(long refreshInterval, Processors processors) throws IOException {
        return new OsInfo(refreshInterval, Runtime.getRuntime().availableProcessors(), processors, Constants.OS_NAME, this.getPrettyName(), Constants.OS_ARCH, Constants.OS_VERSION);
    }

    private String getPrettyName() throws IOException {
        if (Constants.LINUX) {
            Optional maybePrettyNameLine;
            List<String> etcOsReleaseLines = this.readOsRelease();
            List<String> prettyNameLines = etcOsReleaseLines.stream().filter(line -> line.startsWith("PRETTY_NAME")).toList();
            assert (prettyNameLines.size() <= 1) : prettyNameLines;
            Optional<Object> optional = maybePrettyNameLine = prettyNameLines.size() == 1 ? Optional.of(prettyNameLines.get(0)) : Optional.empty();
            if (maybePrettyNameLine.isPresent()) {
                String trimmedPrettyNameLine = ((String)maybePrettyNameLine.get()).trim();
                Matcher matcher = Pattern.compile("PRETTY_NAME=(\"?|'?)?([^\"']+)\\1").matcher(trimmedPrettyNameLine);
                boolean matches = matcher.matches();
                assert (matches) : trimmedPrettyNameLine;
                assert (matcher.groupCount() == 2) : trimmedPrettyNameLine;
                return matcher.group(2);
            }
            return Constants.OS_NAME;
        }
        return Constants.OS_NAME;
    }

    @SuppressForbidden(reason="access /etc/os-release or /usr/lib/os-release or /etc/system-release")
    List<String> readOsRelease() throws IOException {
        if (Files.exists(PathUtils.get((String)"/etc/os-release", (String[])new String[0]), new LinkOption[0])) {
            List<String> lines = Files.readAllLines(PathUtils.get((String)"/etc/os-release", (String[])new String[0]));
            assert (lines != null && !lines.isEmpty());
            return lines;
        }
        if (Files.exists(PathUtils.get((String)"/usr/lib/os-release", (String[])new String[0]), new LinkOption[0])) {
            List<String> lines = Files.readAllLines(PathUtils.get((String)"/usr/lib/os-release", (String[])new String[0]));
            assert (lines != null && !lines.isEmpty());
            return lines;
        }
        if (Files.exists(PathUtils.get((String)"/etc/system-release", (String[])new String[0]), new LinkOption[0])) {
            List<String> lines = Files.readAllLines(PathUtils.get((String)"/etc/system-release", (String[])new String[0]));
            assert (lines != null && lines.size() == 1);
            return Collections.singletonList("PRETTY_NAME=\"" + lines.get(0) + "\"");
        }
        return Collections.emptyList();
    }

    @SuppressForbidden(reason="access /proc/meminfo")
    List<String> readProcMeminfo() throws IOException {
        if (Files.exists(PathUtils.get((String)"/proc/meminfo", (String[])new String[0]), new LinkOption[0])) {
            List<String> lines = Files.readAllLines(PathUtils.get((String)"/proc/meminfo", (String[])new String[0]));
            assert (lines != null && !lines.isEmpty());
            return lines;
        }
        return Collections.emptyList();
    }

    long getTotalMemFromProcMeminfo() throws IOException {
        List<String> meminfoLines = this.readProcMeminfo();
        List<String> memTotalLines = meminfoLines.stream().filter(line -> line.startsWith("MemTotal")).toList();
        assert (memTotalLines.size() <= 1) : memTotalLines;
        if (memTotalLines.size() == 1) {
            int endIdx;
            String memTotalLine = memTotalLines.get(0);
            int beginIdx = memTotalLine.indexOf("MemTotal:");
            if (beginIdx + 9 < (endIdx = memTotalLine.lastIndexOf(" kB"))) {
                String memTotalString = memTotalLine.substring(beginIdx + 9, endIdx).trim();
                try {
                    long memTotalInKb = Long.parseLong(memTotalString);
                    return memTotalInKb * 1024L;
                }
                catch (NumberFormatException e) {
                    logger.warn("Unable to retrieve total memory from meminfo line [" + memTotalLine + "]");
                    return 0L;
                }
            }
            logger.warn("Unable to retrieve total memory from meminfo line [" + memTotalLine + "]");
            return 0L;
        }
        return 0L;
    }

    boolean isDebian8() throws IOException {
        return Constants.LINUX && this.getPrettyName().equals("Debian GNU/Linux 8 (jessie)");
    }

    OsStats.Cgroup getCgroup(boolean isLinux) {
        return isLinux ? this.getCgroup() : null;
    }

    public OsStats osStats() {
        OsStats.Cpu cpu = new OsStats.Cpu(OsProbe.getSystemCpuPercent(), this.getSystemLoadAverage());
        OsStats.Mem mem = new OsStats.Mem(this.getTotalPhysicalMemorySize(), this.getAdjustedTotalMemorySize(), this.getFreePhysicalMemorySize());
        OsStats.Swap swap = new OsStats.Swap(this.getTotalSwapSpaceSize(), this.getFreeSwapSpaceSize());
        OsStats.Cgroup cgroup = this.getCgroup(Constants.LINUX);
        return new OsStats(System.currentTimeMillis(), cpu, mem, swap, cgroup);
    }

    private static Method getMethod(String methodName) {
        String className = "com.sun.management.OperatingSystemMXBean";
        try {
            return Class.forName(className).getMethod(methodName, new Class[0]);
        }
        catch (Exception e) {
            logger.debug(() -> "failed to get method [" + methodName + "] from class [" + className + "]", (Throwable)e);
            return null;
        }
    }

    private static class OsProbeHolder {
        private static final OsProbe INSTANCE = new OsProbe();

        private OsProbeHolder() {
        }
    }
}

