"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerRoutes = void 0;
var _configSchema = require("@kbn/config-schema");
var _constants = require("../../../common/constants");
var _types = require("../../types");
var _errors = require("../schema/errors");
var _api_privileges = require("../../constants/api_privileges");
var _utils = require("../schema/utils");
var _handler = require("./handler");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const registerRoutes = router => {
  router.versioned.get({
    path: _constants.FLEET_SERVER_HOST_API_ROUTES.LIST_PATTERN,
    security: {
      authz: {
        requiredPrivileges: [{
          anyRequired: [_api_privileges.FLEET_API_PRIVILEGES.AGENTS.ALL, _api_privileges.FLEET_API_PRIVILEGES.SETTINGS.READ]
        }]
      }
    },
    summary: `Get Fleet Server hosts`,
    options: {
      tags: ['oas-tag:Fleet Server hosts']
    }
  }).addVersion({
    version: _constants.API_VERSIONS.public.v1,
    validate: {
      request: _types.GetAllFleetServerHostRequestSchema,
      response: {
        200: {
          description: 'OK: A successful request.',
          body: () => (0, _utils.ListResponseSchema)(_types.FleetServerHostSchema)
        },
        400: {
          description: 'A bad request.',
          body: _errors.genericErrorResponse
        }
      }
    }
  }, _handler.getAllFleetServerHostsHandler);
  router.versioned.post({
    path: _constants.FLEET_SERVER_HOST_API_ROUTES.CREATE_PATTERN,
    security: {
      authz: {
        requiredPrivileges: [_api_privileges.FLEET_API_PRIVILEGES.SETTINGS.ALL]
      }
    },
    summary: `Create a Fleet Server host`,
    options: {
      tags: ['oas-tag:Fleet Server hosts']
    }
  }).addVersion({
    version: _constants.API_VERSIONS.public.v1,
    validate: {
      request: _types.PostFleetServerHostRequestSchema,
      response: {
        200: {
          description: 'OK: A successful request.',
          body: () => _types.FleetServerHostResponseSchema
        },
        400: {
          description: 'A bad request.',
          body: _errors.genericErrorResponse
        }
      }
    }
  }, _handler.postFleetServerHost);
  router.versioned.get({
    path: _constants.FLEET_SERVER_HOST_API_ROUTES.INFO_PATTERN,
    security: {
      authz: {
        requiredPrivileges: [_api_privileges.FLEET_API_PRIVILEGES.SETTINGS.READ]
      }
    },
    summary: `Get a Fleet Server host`,
    description: `Get a Fleet Server host by ID.`,
    options: {
      tags: ['oas-tag:Fleet Server hosts']
    }
  }).addVersion({
    version: _constants.API_VERSIONS.public.v1,
    validate: {
      request: _types.GetOneFleetServerHostRequestSchema,
      response: {
        200: {
          description: 'OK: A successful request.',
          body: () => _types.FleetServerHostResponseSchema
        },
        400: {
          description: 'A bad request.',
          body: _errors.genericErrorResponse
        }
      }
    }
  }, _handler.getFleetServerHostHandler);
  router.versioned.delete({
    path: _constants.FLEET_SERVER_HOST_API_ROUTES.DELETE_PATTERN,
    security: {
      authz: {
        requiredPrivileges: [_api_privileges.FLEET_API_PRIVILEGES.SETTINGS.ALL]
      }
    },
    summary: `Delete a Fleet Server host`,
    description: `Delete a Fleet Server host by ID.`,
    options: {
      tags: ['oas-tag:Fleet Server hosts']
    }
  }).addVersion({
    version: _constants.API_VERSIONS.public.v1,
    validate: {
      request: _types.GetOneFleetServerHostRequestSchema,
      response: {
        200: {
          description: 'OK: A successful request.',
          body: () => _configSchema.schema.object({
            id: _configSchema.schema.string()
          })
        },
        400: {
          description: 'A bad request.',
          body: _errors.genericErrorResponse
        }
      }
    }
  }, _handler.deleteFleetServerHostHandler);
  router.versioned.put({
    path: _constants.FLEET_SERVER_HOST_API_ROUTES.UPDATE_PATTERN,
    security: {
      authz: {
        requiredPrivileges: [_api_privileges.FLEET_API_PRIVILEGES.SETTINGS.ALL]
      }
    },
    summary: `Update a Fleet Server host`,
    description: `Update a Fleet Server host by ID.`,
    options: {
      tags: ['oas-tag:Fleet Server hosts']
    }
  }).addVersion({
    version: _constants.API_VERSIONS.public.v1,
    validate: {
      request: _types.PutFleetServerHostRequestSchema,
      response: {
        200: {
          description: 'OK: A successful request.',
          body: () => _types.FleetServerHostResponseSchema
        },
        400: {
          description: 'A bad request.',
          body: _errors.genericErrorResponse
        }
      }
    }
  }, _handler.putFleetServerHostHandler);
};
exports.registerRoutes = registerRoutes;