"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.DocumentsDataWriter = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class DocumentsDataWriter {
  constructor(options) {
    (0, _defineProperty2.default)(this, "bulk", async params => {
      try {
        var _params$documentsToCr, _params$documentsToUp, _params$documentsToDe;
        if (!((_params$documentsToCr = params.documentsToCreate) !== null && _params$documentsToCr !== void 0 && _params$documentsToCr.length) && !((_params$documentsToUp = params.documentsToUpdate) !== null && _params$documentsToUp !== void 0 && _params$documentsToUp.length) && !((_params$documentsToDe = params.documentsToDelete) !== null && _params$documentsToDe !== void 0 && _params$documentsToDe.length)) {
          return {
            errors: [],
            docs_created: [],
            docs_deleted: [],
            docs_updated: [],
            took: 0
          };
        }
        const {
          errors,
          items,
          took
        } = await this.options.esClient.bulk({
          refresh: 'wait_for',
          body: await this.buildBulkOperations(params)
        }, {
          // Increasing timeout to 2min as KB docs were failing to load after 30s
          requestTimeout: 120000
        });
        return {
          errors: errors ? this.formatErrorsResponse(items) : [],
          docs_created: items.filter(item => {
            var _item$create, _item$create2;
            return ((_item$create = item.create) === null || _item$create === void 0 ? void 0 : _item$create.status) === 201 || ((_item$create2 = item.create) === null || _item$create2 === void 0 ? void 0 : _item$create2.status) === 200;
          }).map(item => {
            var _item$create3;
            return (_item$create3 = item.create) === null || _item$create3 === void 0 ? void 0 : _item$create3._id;
          }),
          docs_deleted: items.filter(item => {
            var _item$delete, _item$delete2;
            return ((_item$delete = item.delete) === null || _item$delete === void 0 ? void 0 : _item$delete.status) === 201 || ((_item$delete2 = item.delete) === null || _item$delete2 === void 0 ? void 0 : _item$delete2.status) === 200;
          }).map(item => {
            var _item$delete3;
            return (_item$delete3 = item.delete) === null || _item$delete3 === void 0 ? void 0 : _item$delete3._id;
          }),
          docs_updated: items.filter(item => {
            var _item$update, _item$update2;
            return ((_item$update = item.update) === null || _item$update === void 0 ? void 0 : _item$update.status) === 201 || ((_item$update2 = item.update) === null || _item$update2 === void 0 ? void 0 : _item$update2.status) === 200;
          }).map(item => {
            var _item$update3, _item$update3$get;
            return (_item$update3 = item.update) === null || _item$update3 === void 0 ? void 0 : (_item$update3$get = _item$update3.get) === null || _item$update3$get === void 0 ? void 0 : _item$update3$get._source;
          }),
          took
        };
      } catch (e) {
        this.options.logger.error(`Error bulk actions for documents: ${e.message}`);
        return {
          errors: [{
            message: e.message,
            document: {
              id: ''
            }
          }],
          docs_created: [],
          docs_deleted: [],
          docs_updated: [],
          took: 0
        };
      }
    });
    (0, _defineProperty2.default)(this, "getFilterByUser", authenticatedUser => ({
      filter: {
        bool: {
          should: [{
            bool: {
              must_not: {
                nested: {
                  path: 'users',
                  query: {
                    exists: {
                      field: 'users'
                    }
                  }
                }
              }
            }
          }, {
            nested: {
              path: 'users',
              query: {
                bool: {
                  should: [
                  // Match on users.id if profile_uid exists
                  ...(authenticatedUser.profile_uid ? [{
                    term: {
                      'users.id': authenticatedUser.profile_uid
                    }
                  }] : []),
                  // Always try to match on users.name
                  {
                    term: {
                      'users.name': authenticatedUser.username
                    }
                  }],
                  minimum_should_match: 1
                }
              }
            }
          }]
        }
      }
    }));
    (0, _defineProperty2.default)(this, "getFilterByConversationUser", authenticatedUser => ({
      filter: {
        bool: {
          should: [{
            bool: {
              must: [{
                exists: {
                  field: 'created_by'
                }
              }, {
                bool: {
                  should: [
                  // Match on users.id if profile_uid exists
                  ...(authenticatedUser.profile_uid ? [{
                    term: {
                      'created_by.id': authenticatedUser.profile_uid
                    }
                  }] : []),
                  // Always try to match on users.name
                  {
                    term: {
                      'created_by.name': authenticatedUser.username
                    }
                  }],
                  minimum_should_match: 1
                }
              }]
            }
          }, {
            bool: {
              must: [
              // legacy conversation without created_by field have users array we can match against
              {
                bool: {
                  must_not: [{
                    exists: {
                      field: 'created_by'
                    }
                  }]
                }
              }, {
                nested: {
                  path: 'users',
                  query: {
                    bool: {
                      should: [
                      // Match on users.id if profile_uid exists
                      ...(authenticatedUser.profile_uid ? [{
                        term: {
                          'users.id': authenticatedUser.profile_uid
                        }
                      }] : []),
                      // Always try to match on users.name
                      {
                        term: {
                          'users.name': authenticatedUser.username
                        }
                      }],
                      minimum_should_match: 1
                    }
                  }
                }
              }]
            }
          }],
          minimum_should_match: 1
        }
      }
    }));
    (0, _defineProperty2.default)(this, "getUpdateDocumentsQuery", async (documentsToUpdate, getUpdateScript, authenticatedUser) => {
      const updatedAt = new Date().toISOString();
      const isConversationUpdate = this.options.index.includes('.kibana-elastic-ai-assistant-conversation');
      const responseToUpdate = await this.options.esClient.search({
        query: {
          bool: {
            must: [{
              bool: {
                should: [{
                  ids: {
                    values: documentsToUpdate === null || documentsToUpdate === void 0 ? void 0 : documentsToUpdate.map(c => c.id)
                  }
                }]
              }
            }],
            ...(authenticatedUser ? isConversationUpdate ? this.getFilterByConversationUser(authenticatedUser) : this.getFilterByUser(authenticatedUser) : {})
          }
        },
        _source: false,
        ignore_unavailable: true,
        index: this.options.index,
        seq_no_primary_term: true,
        size: 1000
      });
      const availableDocumentsToUpdate = documentsToUpdate.filter(c => responseToUpdate === null || responseToUpdate === void 0 ? void 0 : responseToUpdate.hits.hits.find(ac => ac._id === c.id));
      return availableDocumentsToUpdate.flatMap(document => {
        var _responseToUpdate$hit;
        return [{
          update: {
            _id: document.id,
            _index: responseToUpdate === null || responseToUpdate === void 0 ? void 0 : (_responseToUpdate$hit = responseToUpdate.hits.hits.find(c => c._id === document.id)) === null || _responseToUpdate$hit === void 0 ? void 0 : _responseToUpdate$hit._index,
            _source: true,
            retry_on_conflict: 3
          }
        }, getUpdateScript(document, updatedAt)];
      });
    });
    (0, _defineProperty2.default)(this, "getDeleteDocumentsQuery", async (documentsToDelete, authenticatedUser) => {
      var _responseToDelete$hit;
      const isConversationUpdate = this.options.index.includes('.kibana-elastic-ai-assistant-conversation');
      const responseToDelete = await this.options.esClient.search({
        query: {
          bool: {
            must: [{
              bool: {
                should: [{
                  ids: {
                    values: documentsToDelete
                  }
                }]
              }
            }],
            ...(authenticatedUser ? isConversationUpdate ? this.getFilterByConversationUser(authenticatedUser) : this.getFilterByUser(authenticatedUser) : {})
          }
        },
        _source: false,
        ignore_unavailable: true,
        index: this.options.index,
        seq_no_primary_term: true,
        size: 1000
      });
      return (_responseToDelete$hit = responseToDelete === null || responseToDelete === void 0 ? void 0 : responseToDelete.hits.hits.map(c => ({
        delete: {
          _id: c._id,
          _index: c._index
        }
      }))) !== null && _responseToDelete$hit !== void 0 ? _responseToDelete$hit : [];
    });
    (0, _defineProperty2.default)(this, "buildBulkOperations", async params => {
      const documentCreateBody = params.documentsToCreate ? params.documentsToCreate.flatMap(document => [
      // Do not pre-gen _id for bulk create operations to avoid `version_conflict_engine_exception`
      {
        create: {
          _index: this.options.index
        }
      }, document]) : [];
      const documentDeletedBody = params.documentsToDelete && params.documentsToDelete.length > 0 ? await this.getDeleteDocumentsQuery(params.documentsToDelete, params.authenticatedUser) : [];
      const documentUpdatedBody = params.documentsToUpdate && params.documentsToUpdate.length > 0 && params.getUpdateScript ? await this.getUpdateDocumentsQuery(params.documentsToUpdate, params.getUpdateScript, params.authenticatedUser) : [];
      return [...documentCreateBody, ...documentUpdatedBody, ...documentDeletedBody];
    });
    (0, _defineProperty2.default)(this, "formatErrorsResponse", items => {
      return items.map(item => {
        var _item$create4, _item$create$error, _item$update4, _item$update$error, _item$delete4, _item$delete5, _item$delete5$error, _item$delete6, _item$delete7;
        return (_item$create4 = item.create) !== null && _item$create4 !== void 0 && _item$create4.error ? {
          message: (_item$create$error = item.create.error) === null || _item$create$error === void 0 ? void 0 : _item$create$error.reason,
          status: item.create.status,
          document: {
            id: item.create._id
          }
        } : (_item$update4 = item.update) !== null && _item$update4 !== void 0 && _item$update4.error ? {
          message: (_item$update$error = item.update.error) === null || _item$update$error === void 0 ? void 0 : _item$update$error.reason,
          status: item.update.status,
          document: {
            id: item.update._id
          }
        } : (_item$delete4 = item.delete) !== null && _item$delete4 !== void 0 && _item$delete4.error ? {
          message: (_item$delete5 = item.delete) === null || _item$delete5 === void 0 ? void 0 : (_item$delete5$error = _item$delete5.error) === null || _item$delete5$error === void 0 ? void 0 : _item$delete5$error.reason,
          status: (_item$delete6 = item.delete) === null || _item$delete6 === void 0 ? void 0 : _item$delete6.status,
          document: {
            id: (_item$delete7 = item.delete) === null || _item$delete7 === void 0 ? void 0 : _item$delete7._id
          }
        } : undefined;
      }).filter(e => e !== undefined);
    });
    this.options = options;
  }
}
exports.DocumentsDataWriter = DocumentsDataWriter;