"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ActionExecutor = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _server = require("@kbn/core/server");
var _lodash = require("lodash");
var _saferLodashSet = require("@kbn/safer-lodash-set");
var _apmUtils = require("@kbn/apm-utils");
var _server2 = require("@kbn/event-log-plugin/server");
var _server3 = require("@kbn/task-manager-plugin/server");
var _task_running = require("@kbn/task-manager-plugin/server/task_running");
var _event_based_telemetry = require("./event_based_telemetry");
var _connector_usage_collector = require("../usage/connector_usage_collector");
var _gen_ai_token_tracking = require("./token_tracking/gen_ai_token_tracking");
var _validate_with_schema = require("./validate_with_schema");
var _types = require("../types");
var _event_log = require("../constants/event_log");
var _create_action_event_log_record_object = require("./create_action_event_log_record_object");
var _action_execution_error = require("./errors/action_execution_error");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

// 1,000,000 nanoseconds in 1 millisecond
const Millis2Nanos = 1000 * 1000;
class ActionExecutor {
  constructor({
    isESOCanEncrypt,
    connectorRateLimiter
  }) {
    (0, _defineProperty2.default)(this, "isInitialized", false);
    (0, _defineProperty2.default)(this, "actionExecutorContext", void 0);
    (0, _defineProperty2.default)(this, "isESOCanEncrypt", void 0);
    (0, _defineProperty2.default)(this, "connectorRateLimiter", void 0);
    (0, _defineProperty2.default)(this, "actionInfo", void 0);
    this.isESOCanEncrypt = isESOCanEncrypt;
    this.connectorRateLimiter = connectorRateLimiter;
  }
  initialize(actionExecutorContext) {
    if (this.isInitialized) {
      throw new Error('ActionExecutor already initialized');
    }
    this.isInitialized = true;
    this.actionExecutorContext = actionExecutorContext;
  }
  async execute({
    actionExecutionId,
    actionId,
    consumer,
    executionId,
    request,
    params,
    relatedSavedObjects,
    source,
    taskInfo
  }) {
    const {
      actionTypeRegistry,
      getActionsAuthorizationWithRequest,
      getServices,
      security,
      spaces
    } = this.actionExecutorContext;
    const services = getServices(request);
    const spaceId = spaces && spaces.getSpaceId(request);
    const namespace = spaceId && spaceId !== 'default' ? {
      namespace: spaceId
    } : {};
    const authorization = getActionsAuthorizationWithRequest(request);
    const currentUser = security === null || security === void 0 ? void 0 : security.authc.getCurrentUser(request);
    return await this.executeHelper({
      actionExecutionId,
      actionId,
      consumer,
      currentUser,
      checkCanExecuteFn: async connectorTypeId => {
        /**
         * Ensures correct permissions for execution and
         * performs authorization checks for system actions.
         * It will thrown an error in case of failure.
         */
        await ensureAuthorizedToExecute({
          params,
          actionId,
          actionTypeId: connectorTypeId,
          actionTypeRegistry,
          authorization,
          source: source === null || source === void 0 ? void 0 : source.type
        });
      },
      executeLabel: `execute_action`,
      executionId,
      namespace,
      params,
      relatedSavedObjects,
      request,
      services,
      source,
      spaceId,
      taskInfo
    });
  }
  async executeUnsecured({
    actionExecutionId,
    actionId,
    params,
    relatedSavedObjects,
    spaceId,
    source
  }) {
    const {
      actionTypeRegistry,
      getUnsecuredServices
    } = this.actionExecutorContext;
    const services = getUnsecuredServices();
    const namespace = spaceId && spaceId !== 'default' ? {
      namespace: spaceId
    } : {};
    return await this.executeHelper({
      actionExecutionId,
      actionId,
      checkCanExecuteFn: async connectorTypeId => {
        let errorMessage = null;
        if (_types.UNALLOWED_FOR_UNSECURE_EXECUTION_CONNECTOR_TYPE_IDS.includes(connectorTypeId)) {
          errorMessage = `Cannot execute unsecured "${connectorTypeId}" action - execution of this type is not allowed`;
        }

        // We don't allow execute system actions in unsecured manner because they require a request
        if (actionTypeRegistry.isSystemActionType(connectorTypeId)) {
          errorMessage = `Cannot execute unsecured system action`;
        }
        if (errorMessage) {
          throw new _action_execution_error.ActionExecutionError(errorMessage, _action_execution_error.ActionExecutionErrorReason.Authorization, {
            actionId,
            status: 'error',
            message: errorMessage,
            retry: false,
            errorSource: _server3.TaskErrorSource.USER
          });
        }
      },
      executeLabel: `execute_unsecured_action`,
      namespace,
      params,
      relatedSavedObjects,
      services,
      source,
      spaceId
    });
  }
  async logCancellation({
    actionId,
    request,
    relatedSavedObjects,
    source,
    executionId,
    taskInfo,
    consumer,
    actionExecutionId
  }) {
    var _this$actionInfo$name;
    const {
      spaces,
      eventLogger
    } = this.actionExecutorContext;
    const spaceId = spaces && spaces.getSpaceId(request);
    const namespace = spaceId && spaceId !== 'default' ? {
      namespace: spaceId
    } : {};
    if (!this.actionInfo || this.actionInfo.actionId !== actionId) {
      this.actionInfo = await this.getActionInfoInternal(actionId, namespace.namespace);
    }
    const task = taskInfo ? {
      task: {
        scheduled: taskInfo.scheduled.toISOString(),
        scheduleDelay: Millis2Nanos * (Date.now() - taskInfo.scheduled.getTime())
      }
    } : {};
    // Write event log entry
    const event = (0, _create_action_event_log_record_object.createActionEventLogRecordObject)({
      actionId,
      consumer,
      action: _event_log.EVENT_LOG_ACTIONS.executeTimeout,
      message: `action: ${this.actionInfo.actionTypeId}:${actionId}: '${(_this$actionInfo$name = this.actionInfo.name) !== null && _this$actionInfo$name !== void 0 ? _this$actionInfo$name : ''}' execution cancelled due to timeout - exceeded default timeout of "5m"`,
      ...namespace,
      ...task,
      executionId,
      spaceId,
      savedObjects: [{
        type: 'action',
        id: actionId,
        typeId: this.actionInfo.actionTypeId,
        relation: _server2.SAVED_OBJECT_REL_PRIMARY
      }],
      relatedSavedObjects,
      actionExecutionId,
      isInMemory: this.actionInfo.isInMemory,
      ...(source ? {
        source
      } : {}),
      actionTypeId: this.actionInfo.actionTypeId
    });
    eventLogger.logEvent(event);
  }
  async getActionInfoInternal(actionId, namespace) {
    const {
      encryptedSavedObjectsClient,
      inMemoryConnectors
    } = this.actionExecutorContext;

    // check to see if it's in memory connector first
    const inMemoryAction = inMemoryConnectors.find(inMemoryConnector => inMemoryConnector.id === actionId);
    if (inMemoryAction) {
      return {
        actionTypeId: inMemoryAction.actionTypeId,
        name: inMemoryAction.name,
        config: inMemoryAction.config,
        secrets: inMemoryAction.secrets,
        actionId,
        isInMemory: true,
        rawAction: {
          ...inMemoryAction,
          isMissingSecrets: false
        }
      };
    }
    if (!this.isESOCanEncrypt) {
      throw (0, _server3.createTaskRunError)(new Error(`Unable to execute action because the Encrypted Saved Objects plugin is missing encryption key. Please set xpack.encryptedSavedObjects.encryptionKey in the kibana.yml or use the bin/kibana-encryption-keys command.`), _server3.TaskErrorSource.FRAMEWORK);
    }
    try {
      const rawAction = await encryptedSavedObjectsClient.getDecryptedAsInternalUser('action', actionId, {
        namespace: namespace === 'default' ? undefined : namespace
      });
      const {
        attributes: {
          secrets,
          actionTypeId,
          config,
          name
        }
      } = rawAction;
      return {
        actionTypeId,
        name,
        config,
        secrets,
        actionId,
        rawAction: rawAction.attributes
      };
    } catch (e) {
      if (_server.SavedObjectsErrorHelpers.isNotFoundError(e)) {
        throw (0, _server3.createTaskRunError)(e, _server3.TaskErrorSource.USER);
      }
      throw (0, _server3.createTaskRunError)(e, _server3.TaskErrorSource.FRAMEWORK);
    }
  }
  async executeHelper({
    actionExecutionId,
    actionId,
    consumer,
    currentUser,
    checkCanExecuteFn,
    executeLabel,
    executionId,
    namespace,
    params,
    relatedSavedObjects,
    request,
    services,
    source,
    spaceId,
    taskInfo
  }) {
    if (!this.isInitialized) {
      throw new Error('ActionExecutor not initialized');
    }
    return (0, _apmUtils.withSpan)({
      name: executeLabel,
      type: 'actions',
      labels: {
        actions_connector_id: actionId
      }
    }, async span => {
      const {
        actionTypeRegistry,
        analyticsService,
        eventLogger
      } = this.actionExecutorContext;
      const actionInfo = await this.getActionInfoInternal(actionId, namespace.namespace);
      const {
        actionTypeId,
        name,
        config,
        secrets
      } = actionInfo;
      const loggerId = actionTypeId.startsWith('.') ? actionTypeId.substring(1) : actionTypeId;
      const logger = this.actionExecutorContext.logger.get(loggerId);
      const connectorUsageCollector = new _connector_usage_collector.ConnectorUsageCollector({
        logger,
        connectorId: actionId
      });
      if (!this.actionInfo || this.actionInfo.actionId !== actionId) {
        this.actionInfo = actionInfo;
      }
      if (this.connectorRateLimiter.isRateLimited(actionTypeId)) {
        return {
          actionId,
          status: 'error',
          message: `Action execution rate limit exceeded for connector: ${actionTypeId}`,
          retry: true,
          errorSource: _server3.TaskErrorSource.USER
        };
      }
      if (!actionTypeRegistry.isActionExecutable(actionId, actionTypeId, {
        notifyUsage: true
      })) {
        try {
          actionTypeRegistry.ensureActionTypeEnabled(actionTypeId);
        } catch (e) {
          throw (0, _server3.createTaskRunError)(e, _server3.TaskErrorSource.FRAMEWORK);
        }
      }
      const actionType = actionTypeRegistry.get(actionTypeId);
      const configurationUtilities = actionTypeRegistry.getUtils();
      let validatedParams;
      let validatedConfig;
      let validatedSecrets;
      try {
        const validationResult = validateAction({
          actionId,
          actionType,
          params,
          config,
          secrets,
          taskInfo
        }, {
          configurationUtilities
        });
        validatedParams = validationResult.validatedParams;
        validatedConfig = validationResult.validatedConfig;
        validatedSecrets = validationResult.validatedSecrets;
      } catch (err) {
        return err.result;
      }
      if (span) {
        span.name = `${executeLabel} ${actionTypeId}`;
        span.addLabels({
          actions_connector_type_id: actionTypeId
        });
      }
      const actionLabel = `${actionTypeId}:${actionId}: ${name}`;
      logger.debug(`executing action ${actionLabel}`);
      const task = taskInfo ? {
        task: {
          scheduled: taskInfo.scheduled.toISOString(),
          scheduleDelay: Millis2Nanos * (Date.now() - taskInfo.scheduled.getTime())
        }
      } : {};
      const event = (0, _create_action_event_log_record_object.createActionEventLogRecordObject)({
        actionId,
        action: _event_log.EVENT_LOG_ACTIONS.execute,
        consumer,
        ...namespace,
        ...task,
        executionId,
        spaceId,
        savedObjects: [{
          type: 'action',
          id: actionId,
          typeId: actionTypeId,
          relation: _server2.SAVED_OBJECT_REL_PRIMARY
        }],
        relatedSavedObjects,
        name,
        actionExecutionId,
        isInMemory: this.actionInfo.isInMemory,
        ...(source ? {
          source
        } : {}),
        actionTypeId
      });
      eventLogger.startTiming(event);
      const startEvent = (0, _lodash.cloneDeep)({
        ...event,
        event: {
          ...event.event,
          action: _event_log.EVENT_LOG_ACTIONS.executeStart
        },
        message: `action started: ${actionLabel}`
      });
      eventLogger.logEvent(startEvent);
      let rawResult;
      try {
        if (checkCanExecuteFn) {
          await checkCanExecuteFn(actionTypeId);
        }
        rawResult = await actionType.executor({
          actionId,
          services,
          params: validatedParams,
          config: validatedConfig,
          secrets: validatedSecrets,
          taskInfo,
          configurationUtilities,
          logger,
          source,
          ...(actionType.isSystemActionType ? {
            request
          } : {}),
          connectorUsageCollector
        });
        if (rawResult && rawResult.status === 'error') {
          rawResult.errorSource = _server3.TaskErrorSource.USER;
        }
      } catch (err) {
        const errorSource = (0, _task_running.getErrorSource)(err) || _server3.TaskErrorSource.FRAMEWORK;
        if (err.reason === _action_execution_error.ActionExecutionErrorReason.Authorization) {
          rawResult = err.result;
        } else {
          rawResult = {
            actionId,
            status: 'error',
            message: 'an error occurred while running the action',
            serviceMessage: err.message,
            error: err,
            retry: true,
            errorSource
          };
        }
      }
      this.connectorRateLimiter.log(actionTypeId);

      // allow null-ish return to indicate success
      const result = rawResult || {
        actionId,
        status: 'ok'
      };
      event.event = event.event || {};
      const {
        error,
        ...resultWithoutError
      } = result;
      function completeEventLogging() {
        eventLogger.stopTiming(event);
        event.user = event.user || {};
        event.user.name = currentUser === null || currentUser === void 0 ? void 0 : currentUser.username;
        event.user.id = currentUser === null || currentUser === void 0 ? void 0 : currentUser.profile_uid;
        if (currentUser !== null && currentUser !== void 0 && currentUser.api_key) {
          var _currentUser$api_key, _currentUser$api_key2;
          event.kibana.user_api_key = {
            name: (_currentUser$api_key = currentUser.api_key) === null || _currentUser$api_key === void 0 ? void 0 : _currentUser$api_key.name,
            id: (_currentUser$api_key2 = currentUser.api_key) === null || _currentUser$api_key2 === void 0 ? void 0 : _currentUser$api_key2.id
          };
        }
        (0, _saferLodashSet.set)(event, 'kibana.action.execution.usage.request_body_bytes', connectorUsageCollector.getRequestBodyByte());
        if (result.status === 'ok') {
          span === null || span === void 0 ? void 0 : span.setOutcome('success');
          event.event.outcome = 'success';
          event.message = `action executed: ${actionLabel}`;
        } else if (result.status === 'error') {
          span === null || span === void 0 ? void 0 : span.setOutcome('failure');
          event.event.outcome = 'failure';
          event.message = `action execution failure: ${actionLabel}`;
          event.error = event.error || {};
          event.error.message = actionErrorToMessage(result);
          if (result.error) {
            logger.error(result.error, {
              tags: [actionTypeId, actionId, 'action-run-failed', `${result.errorSource}-error`],
              error: {
                stack_trace: result.error.stack
              }
            });
          }
          logger.warn(`action execution failure: ${actionLabel}: ${event.error.message}`);
        } else {
          span === null || span === void 0 ? void 0 : span.setOutcome('failure');
          event.event.outcome = 'failure';
          event.message = `action execution returned unexpected result: ${actionLabel}: "${result.status}"`;
          event.error = event.error || {};
          event.error.message = 'action execution returned unexpected result';
          logger.warn(`action execution failure: ${actionLabel}: returned unexpected result "${result.status}"`);
        }
        eventLogger.logEvent(event);
      }

      // start genai extension
      if (result.status === 'ok' && (0, _gen_ai_token_tracking.shouldTrackGenAiToken)(actionTypeId, `${validatedParams.subAction}`)) {
        (0, _gen_ai_token_tracking.getGenAiTokenTracking)({
          actionTypeId,
          logger,
          result,
          validatedParams
        }).then(tokenTracking => {
          if (tokenTracking != null) {
            var _tokenTracking$total_, _tokenTracking$prompt, _tokenTracking$comple, _tokenTracking$total_2, _tokenTracking$prompt2, _tokenTracking$comple2, _tokenTracking$teleme, _tokenTracking$teleme2;
            (0, _saferLodashSet.set)(event, 'kibana.action.execution.gen_ai.usage', {
              total_tokens: (_tokenTracking$total_ = tokenTracking.total_tokens) !== null && _tokenTracking$total_ !== void 0 ? _tokenTracking$total_ : 0,
              prompt_tokens: (_tokenTracking$prompt = tokenTracking.prompt_tokens) !== null && _tokenTracking$prompt !== void 0 ? _tokenTracking$prompt : 0,
              completion_tokens: (_tokenTracking$comple = tokenTracking.completion_tokens) !== null && _tokenTracking$comple !== void 0 ? _tokenTracking$comple : 0
            });
            analyticsService.reportEvent(_event_based_telemetry.GEN_AI_TOKEN_COUNT_EVENT.eventType, {
              actionTypeId,
              total_tokens: (_tokenTracking$total_2 = tokenTracking.total_tokens) !== null && _tokenTracking$total_2 !== void 0 ? _tokenTracking$total_2 : 0,
              prompt_tokens: (_tokenTracking$prompt2 = tokenTracking.prompt_tokens) !== null && _tokenTracking$prompt2 !== void 0 ? _tokenTracking$prompt2 : 0,
              completion_tokens: (_tokenTracking$comple2 = tokenTracking.completion_tokens) !== null && _tokenTracking$comple2 !== void 0 ? _tokenTracking$comple2 : 0,
              aggregateBy: tokenTracking === null || tokenTracking === void 0 ? void 0 : (_tokenTracking$teleme = tokenTracking.telemetry_metadata) === null || _tokenTracking$teleme === void 0 ? void 0 : _tokenTracking$teleme.aggregateBy,
              pluginId: tokenTracking === null || tokenTracking === void 0 ? void 0 : (_tokenTracking$teleme2 = tokenTracking.telemetry_metadata) === null || _tokenTracking$teleme2 === void 0 ? void 0 : _tokenTracking$teleme2.pluginId,
              ...(actionTypeId === '.gen-ai' && (config === null || config === void 0 ? void 0 : config.apiProvider) != null ? {
                provider: config === null || config === void 0 ? void 0 : config.apiProvider
              } : {}),
              ...((config === null || config === void 0 ? void 0 : config.defaultModel) != null ? {
                model: config === null || config === void 0 ? void 0 : config.defaultModel
              } : {})
            });
          }
        }).catch(err => {
          logger.error('Failed to calculate tokens from streaming response');
          logger.error(err);
        }).finally(() => {
          completeEventLogging();
        });
        return resultWithoutError;
      }
      // end genai extension

      completeEventLogging();
      return resultWithoutError;
    });
  }
}
exports.ActionExecutor = ActionExecutor;
function actionErrorToMessage(result) {
  let message = result.message || 'unknown error running action';
  if (result.serviceMessage) {
    message = `${message}: ${result.serviceMessage}`;
  }
  if (result.retry instanceof Date) {
    message = `${message}; retry at ${result.retry.toISOString()}`;
  } else if (result.retry) {
    message = `${message}; retry: ${JSON.stringify(result.retry)}`;
  }
  return message;
}
function validateAction({
  actionId,
  actionType,
  params,
  config,
  secrets,
  taskInfo
}, validatorServices) {
  let validatedParams;
  let validatedConfig;
  let validatedSecrets;
  try {
    validatedParams = (0, _validate_with_schema.validateParams)(actionType, params, validatorServices);
  } catch (err) {
    throw new _action_execution_error.ActionExecutionError(err.message, _action_execution_error.ActionExecutionErrorReason.Validation, {
      actionId,
      status: 'error',
      message: err.message,
      retry: !!taskInfo,
      errorSource: _server3.TaskErrorSource.USER
    });
  }
  try {
    var _actionType$validate;
    validatedConfig = (0, _validate_with_schema.validateConfig)(actionType, config, validatorServices);
    validatedSecrets = (0, _validate_with_schema.validateSecrets)(actionType, secrets, validatorServices);
    if ((_actionType$validate = actionType.validate) !== null && _actionType$validate !== void 0 && _actionType$validate.connector) {
      (0, _validate_with_schema.validateConnector)(actionType, {
        config,
        secrets
      });
    }
    return {
      validatedParams,
      validatedConfig,
      validatedSecrets
    };
  } catch (err) {
    throw new _action_execution_error.ActionExecutionError(err.message, _action_execution_error.ActionExecutionErrorReason.Validation, {
      actionId,
      status: 'error',
      message: err.message,
      retry: !!taskInfo,
      errorSource: _server3.TaskErrorSource.FRAMEWORK
    });
  }
}
const ensureAuthorizedToExecute = async ({
  actionId,
  actionTypeId,
  params,
  actionTypeRegistry,
  authorization,
  source
}) => {
  try {
    if (actionTypeRegistry.isSystemActionType(actionTypeId) || actionTypeRegistry.hasSubFeature(actionTypeId)) {
      const additionalPrivileges = actionTypeRegistry.getActionKibanaPrivileges(actionTypeId, params, source);
      await authorization.ensureAuthorized({
        operation: 'execute',
        additionalPrivileges,
        actionTypeId
      });
    }
  } catch (error) {
    throw new _action_execution_error.ActionExecutionError(error.message, _action_execution_error.ActionExecutionErrorReason.Authorization, {
      actionId,
      status: 'error',
      message: error.message,
      retry: false,
      errorSource: _server3.TaskErrorSource.USER
    });
  }
};