"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.flattenHit = flattenHit;
exports.tabifyDocs = exports.getFlattenedFieldsComparator = void 0;
var _lodash = require("lodash");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

// meta fields we won't merge with our result hit
const EXCLUDED_META_FIELDS = ['_type', '_source'];

// This is an overwrite of the SearchHit type to add the ignored_field_values.
// Can be removed once the estypes.SearchHit knows about ignored_field_values

function flattenAccum(flat, obj, keyPrefix, indexPattern, params) {
  for (const k in obj) {
    if (!Object.hasOwn(obj, k)) {
      continue;
    }
    const val = obj[k];
    const key = keyPrefix + k;
    const field = indexPattern === null || indexPattern === void 0 ? void 0 : indexPattern.fields.getByName(key);
    if ((params === null || params === void 0 ? void 0 : params.shallow) === false) {
      const isNestedField = (field === null || field === void 0 ? void 0 : field.type) === 'nested';
      if (Array.isArray(val) && !isNestedField) {
        for (let i = 0; i < val.length; i++) {
          const v = val[i];
          if ((0, _lodash.isPlainObject)(v)) {
            flattenAccum(flat, v, key + '.', indexPattern, params);
          }
        }
        continue;
      }
    } else if (flat[key] !== undefined) {
      continue;
    }
    const hasValidMapping = field && field.type !== 'conflict';
    const isValue = !(0, _lodash.isPlainObject)(val);
    if (hasValidMapping || isValue) {
      if (!flat[key]) {
        flat[key] = val;
      } else if (Array.isArray(flat[key])) {
        flat[key].push(val);
      } else {
        flat[key] = [flat[key], val];
      }
      continue;
    }
    flattenAccum(flat, val, key + '.', indexPattern, params);
  }
}

/**
 * Flattens an individual hit (from an ES response) into an object. This will
 * create flattened field names, like `user.name`.
 *
 * @param hit The hit from an ES reponse's hits.hits[]
 * @param indexPattern The index pattern for the requested index if available.
 * @param params Parameters how to flatten the hit
 */
function flattenHit(hit, indexPattern, params) {
  const flat = {};
  flattenAccum(flat, hit.fields || {}, '', indexPattern, params);
  if ((params === null || params === void 0 ? void 0 : params.source) !== false && hit._source) {
    flattenAccum(flat, hit._source, '', indexPattern, params);
  } else if (params !== null && params !== void 0 && params.includeIgnoredValues && hit.ignored_field_values) {
    // If enabled merge the ignored_field_values into the flattened hit. This will
    // merge values that are not actually indexed by ES (i.e. ignored), e.g. because
    // they were above the `ignore_above` limit or malformed for specific types.
    // This API will only contain the values that were actually ignored, i.e. for the same
    // field there might exist another value in the `fields` response, why this logic
    // merged them both together. We do not merge this (even if enabled) in case source has been
    // merged, since we would otherwise duplicate values, since ignore_field_values and _source
    // contain the same values.
    for (const fieldName in hit.ignored_field_values) {
      if (!Object.hasOwn(hit.ignored_field_values, fieldName)) {
        continue;
      }
      const fieldValue = hit.ignored_field_values[fieldName];
      if (flat[fieldName]) {
        // If there was already a value from the fields API, make sure we're merging both together
        if (Array.isArray(flat[fieldName])) {
          flat[fieldName] = [...flat[fieldName], ...fieldValue];
        } else {
          flat[fieldName] = [flat[fieldName], ...fieldValue];
        }
      } else {
        // If no previous value was assigned we can simply use the value from `ignored_field_values` as it is
        flat[fieldName] = fieldValue;
      }
    }
  }

  // Merge all valid meta fields into the flattened object
  if (indexPattern !== null && indexPattern !== void 0 && indexPattern.metaFields) {
    for (let i = 0; i < (indexPattern === null || indexPattern === void 0 ? void 0 : indexPattern.metaFields.length); i++) {
      const fieldName = indexPattern === null || indexPattern === void 0 ? void 0 : indexPattern.metaFields[i];
      const isExcludedMetaField = EXCLUDED_META_FIELDS.includes(fieldName) || fieldName.charAt(0) !== '_';
      if (!isExcludedMetaField) {
        flat[fieldName] = hit[fieldName];
      }
    }
  }

  // Use a proxy to make sure that keys are always returned in a specific order,
  // so we have a guarantee on the flattened order of keys.
  return makeProxy(flat, indexPattern, params === null || params === void 0 ? void 0 : params.flattenedFieldsComparator);
}
const getFlattenedFieldsComparator = indexPattern => {
  const metaFields = new Set(indexPattern === null || indexPattern === void 0 ? void 0 : indexPattern.metaFields);
  const lowerMap = new Map();
  let aLower;
  let bLower;
  const compareLower = (a, b) => {
    aLower = lowerMap.get(a);
    if (aLower === undefined) {
      aLower = a.toLowerCase();
      lowerMap.set(a, aLower);
    }
    bLower = lowerMap.get(b);
    if (bLower === undefined) {
      bLower = b.toLowerCase();
      lowerMap.set(b, bLower);
    }
    return aLower < bLower ? -1 : aLower > bLower ? 1 : 0;
  };
  return (a, b) => {
    if (typeof a === 'symbol' || typeof b === 'symbol') {
      return 0;
    }
    const aIsMeta = metaFields.has(a);
    const bIsMeta = metaFields.has(b);
    if (aIsMeta && bIsMeta) {
      return compareLower(a, b);
    }
    if (aIsMeta) {
      return 1;
    }
    if (bIsMeta) {
      return -1;
    }
    return compareLower(a, b);
  };
};
exports.getFlattenedFieldsComparator = getFlattenedFieldsComparator;
function makeProxy(flat, indexPattern, flattenedFieldsComparator) {
  let cachedKeys;
  return new Proxy(flat, {
    defineProperty: (...args) => {
      cachedKeys = undefined;
      return Reflect.defineProperty(...args);
    },
    deleteProperty: (...args) => {
      cachedKeys = undefined;
      return Reflect.deleteProperty(...args);
    },
    ownKeys: target => {
      if (!cachedKeys) {
        cachedKeys = Reflect.ownKeys(target).sort(flattenedFieldsComparator !== null && flattenedFieldsComparator !== void 0 ? flattenedFieldsComparator : getFlattenedFieldsComparator(indexPattern));
      }
      return cachedKeys;
    }
  });
}
const tabifyDocs = (esResponse, index, params = {}) => {
  const columns = [];
  const rows = esResponse.hits.hits.map(hit => {
    const flat = flattenHit(hit, index, params);
    if (!flat) {
      return;
    }
    for (const [key, value] of Object.entries(flat)) {
      const field = index === null || index === void 0 ? void 0 : index.fields.getByName(key);
      const fieldName = (field === null || field === void 0 ? void 0 : field.name) || key;
      if (!columns.find(c => c.id === fieldName)) {
        const fieldType = (field === null || field === void 0 ? void 0 : field.type) || typeof value;
        const formatter = field && (index === null || index === void 0 ? void 0 : index.getFormatterForField(field));
        columns.push({
          id: fieldName,
          name: fieldName,
          meta: {
            type: fieldType,
            field: fieldName,
            index: index === null || index === void 0 ? void 0 : index.id,
            params: formatter ? formatter.toJSON() : undefined
          }
        });
      }
    }
    return flat;
  }).filter(hit => hit);
  return {
    type: 'datatable',
    columns,
    rows: rows
  };
};
exports.tabifyDocs = tabifyDocs;