"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.diffCategorization = diffCategorization;
exports.selectResults = selectResults;
exports.stringArraysEqual = stringArraysEqual;
var _common = require("../../../common");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Selects a subset of results for further processing from the given list.
 *
 * The shuffle is deterministic and reproducible, based on the default seed.
 *
 * @param pipelineResults - An array of PipelineResult objects to select from.
 * @param maxSamples - The maximum number of samples to select.
 * @returns An array of PipelineResult objects, containing up to `maxSamples` elements and their indices.
 */
function selectResults(pipelineResults, maxSamples, avoidIndices) {
  const numSamples = Math.min(pipelineResults.length, maxSamples);
  const indices = Array.from({
    length: pipelineResults.length
  }, (_, i) => i).filter(i => !avoidIndices.has(i));
  if (indices.length < numSamples) {
    const avoidIndicesList = Array.from(avoidIndices).sort();
    (0, _common.partialShuffleArray)(avoidIndicesList, 0, numSamples - indices.length);
    avoidIndicesList.length = numSamples - indices.length;
    indices.push(...avoidIndicesList);
  }
  (0, _common.partialShuffleArray)(indices, 0, numSamples);
  indices.length = numSamples;
  return [indices.map(i => pipelineResults[i]), indices];
}

/**
 * Converts a PipelineResult object into its categorization.
 *
 * @param {PipelineResult} result - The result object from the pipeline containing event details.
 * @returns {string[]} An array of strings combining event categories and types. Returns an empty array if event, event.category, or event.type is missing.
 */
function toCategorization(result) {
  const event = result === null || result === void 0 ? void 0 : result.event;
  if (!event || !event.category || !event.type) {
    return [];
  }
  return [...event.category.sort(), ...event.type.sort()];
}

/**
 * Compares two arrays of strings for equality.
 *
 * @param arr1 - The first array of strings to compare.
 * @param arr2 - The second array of strings to compare.
 * @returns the equality predicate
 */
function stringArraysEqual(arr1, arr2) {
  return arr1.length === arr2.length && arr1.every((value, index) => value === arr2[index]);
}

/**
 * Compares two arrays of pipeline results and returns a set of indices where the categorization differs.
 *
 * @param pipelineResults - The current array of pipeline results.
 * @param previousPipelineResults - The previous array of pipeline results to compare against.
 * @returns A set of indices where the pipeline results differ in event category or type.
 */
function diffCategorization(pipelineResults, previousPipelineResults) {
  const diff = Array.from({
    length: pipelineResults.length
  }, (_, i) => i).filter(i => {
    const category1 = toCategorization(pipelineResults[i]);
    const category2 = toCategorization(previousPipelineResults[i]);
    return !stringArraysEqual(category1, category2);
  });
  return new Set(diff);
}