"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.buildRoute = buildRoute;
exports.handle = void 0;
exports.isSafeMethod = isSafeMethod;
exports.validateHapiRequest = validateHapiRequest;
var _coreHttpServer = require("@kbn/core-http-server");
var _configSchema = require("@kbn/config-schema");
var _zod = require("@kbn/zod");
var _request = require("./request");
var _validator = require("./validator");
var _versioned_router = require("./versioned_router");
var _response = require("./response");
var _util = require("./util");
var _security_route_config_validator = require("./security_route_config_validator");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

function isSafeMethod(method) {
  return method === 'get' || method === 'options';
}

/** @interval */

/** @internal */

function buildRoute({
  handler,
  log,
  route,
  router,
  method
}) {
  route = (0, _util.prepareRouteConfigValidation)(route);
  const routeSchemas = routeSchemasFromRouteConfig(route, method);
  return {
    handler: async req => {
      return await handle(req, {
        handler,
        log,
        method,
        route,
        router,
        routeSchemas
      });
    },
    method,
    path: (0, _util.getRouteFullPath)(router.routerPath, route.path),
    options: (0, _util.validOptions)(method, route),
    security: (0, _security_route_config_validator.validRouteSecurity)(route.security, route.options),
    validationSchemas: route.validate,
    isVersioned: false
  };
}

/** @internal */

/** @internal */
function validateHapiRequest(request, {
  routeInfo,
  router,
  log,
  routeSchemas,
  version
}) {
  let kibanaRequest;
  try {
    kibanaRequest = _request.CoreKibanaRequest.from(request, routeSchemas);
    kibanaRequest.apiVersion = version;
  } catch (error) {
    kibanaRequest = _request.CoreKibanaRequest.from(request);
    kibanaRequest.apiVersion = version;
    log.error('400 Bad Request', (0, _util.formatErrorMeta)(400, {
      request,
      error
    }));
    const response = _response.kibanaResponseFactory.badRequest({
      body: error.message,
      headers: isPublicAccessApiRoute(routeInfo) ? (0, _util.getVersionHeader)(_versioned_router.BASE_PUBLIC_VERSION) : undefined
    });
    return {
      error: response
    };
  } finally {
    router.emitPostValidate(kibanaRequest, getPostValidateEventMetadata(kibanaRequest, routeInfo));
  }
  return {
    ok: kibanaRequest
  };
}

/** @internal */
const handle = async (request, {
  router,
  route,
  handler,
  routeSchemas,
  log
}) => {
  var _route$options, _route$options2, _route$options3;
  const {
    error,
    ok: kibanaRequest
  } = validateHapiRequest(request, {
    routeInfo: {
      access: (_route$options = route.options) === null || _route$options === void 0 ? void 0 : _route$options.access,
      httpResource: (_route$options2 = route.options) === null || _route$options2 === void 0 ? void 0 : _route$options2.httpResource,
      deprecated: (_route$options3 = route.options) === null || _route$options3 === void 0 ? void 0 : _route$options3.deprecated
    },
    router,
    log,
    routeSchemas
  });
  if (error) {
    return error;
  }
  const kibanaResponse = await handler(kibanaRequest, _response.kibanaResponseFactory);
  if (isPublicAccessApiRoute(route.options)) {
    (0, _util.injectVersionHeader)(_versioned_router.BASE_PUBLIC_VERSION, kibanaResponse);
  }
  return kibanaResponse;
};
exports.handle = handle;
function isPublicAccessApiRoute({
  access,
  httpResource
} = {}) {
  return !httpResource && access === 'public';
}

/**
 * Create the validation schemas for a route
 *
 * @returns Route schemas if `validate` is specified on the route, otherwise
 * undefined.
 */
function routeSchemasFromRouteConfig(route, routeMethod) {
  // The type doesn't allow `validate` to be undefined, but it can still
  // happen when it's used from JavaScript.
  if (route.validate === undefined) {
    throw new Error(`The [${routeMethod}] at [${route.path}] does not have a 'validate' specified. Use 'false' as the value if you want to bypass validation.`);
  }
  if (route.validate !== false) {
    const validation = (0, _coreHttpServer.getRequestValidation)(route.validate);
    Object.entries(validation).forEach(([key, schema]) => {
      if (!((0, _configSchema.isConfigSchema)(schema) || (0, _zod.isZod)(schema) || typeof schema === 'function')) {
        throw new Error(`Expected a valid validation logic declared with '@kbn/config-schema' package, '@kbn/zod' package or a RouteValidationFunction at key: [${key}].`);
      }
    });
    return _validator.RouteValidator.from(validation);
  }
}
function getPostValidateEventMetadata(request, routeInfo) {
  return {
    deprecated: routeInfo.deprecated,
    isInternalApiRequest: request.isInternalApiRequest,
    isPublicAccess: routeInfo.access === 'public'
  };
}