"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.stateHandlerFactory = void 0;
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getDefaultStreamState = overrides => {
  var _overrides$loaded, _overrides$sampleProb;
  return {
    isRunning: !!overrides.isRunning,
    groupingEnabled: !!overrides.groupingEnabled,
    loaded: (_overrides$loaded = overrides.loaded) !== null && _overrides$loaded !== void 0 ? _overrides$loaded : 0,
    sampleProbability: (_overrides$sampleProb = overrides.sampleProbability) !== null && _overrides$sampleProb !== void 0 ? _overrides$sampleProb : 1,
    shouldStop: !!overrides.shouldStop
  };
};

/**
 * `stateHandlerFactory` takes care of handling the inner state of the stream,
 * for example how much of the stream has been completed, if the stream is running etc.
 * It exposes the state as getter/setter functions, for example `loaded()` will
 * retrieve the current loading state, `loaded(0.5)` will update it.
 */
const stateHandlerFactory = overrides => {
  const state = getDefaultStreamState(overrides);
  function groupingEnabled(d) {
    if (typeof d === 'boolean') {
      state.groupingEnabled = d;
    } else {
      return state.groupingEnabled;
    }
  }
  function loaded(d, replace = true) {
    if (typeof d === 'number') {
      if (replace) {
        state.loaded = Math.round(d * 100) / 100;
      } else {
        state.loaded += Math.round(d * 100) / 100;
      }
    } else {
      return state.loaded;
    }
  }
  function isRunning(d) {
    if (typeof d === 'boolean') {
      state.isRunning = d;
    } else {
      return state.isRunning;
    }
  }
  function sampleProbability(d) {
    if (typeof d === 'number') {
      state.sampleProbability = d;
    } else {
      return state.sampleProbability;
    }
  }
  function shouldStop(d) {
    if (typeof d === 'boolean') {
      state.shouldStop = d;
    } else {
      return state.shouldStop;
    }
  }
  return {
    groupingEnabled,
    isRunning,
    loaded,
    sampleProbability,
    shouldStop
  };
};
exports.stateHandlerFactory = stateHandlerFactory;