"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createSyncMarkersService = void 0;
var _datemath = _interopRequireDefault(require("@kbn/datemath"));
var _saved_objects = require("../../../../saved_objects");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const FIRST_RUN_DEFAULT_RANGE = 'now-1M'; // on first run (update detection), look back over last month

const parseOrThrow = expr => {
  const date = _datemath.default.parse(expr);
  if (!date) throw new Error(`Invalid date math: ${expr}`);
  return date.toDate().toISOString();
};
const createSyncMarkersService = (dataClient, soClient) => {
  const {
    deps
  } = dataClient;
  const monitoringIndexSourceClient = new _saved_objects.MonitoringEntitySourceDescriptorClient({
    soClient,
    namespace: deps.namespace
  });
  const updateLastProcessedMarker = async (source, lastProcessedMarker) => {
    await monitoringIndexSourceClient.updateLastProcessedMarker(source, lastProcessedMarker);
  };
  const getLastProcessedMarker = async source => {
    const lastProcessedMarker = await monitoringIndexSourceClient.getLastProcessedMarker(source);
    // if last processed doesn't exist, return default of now-1M. This is assumed first run for this source.
    if (!lastProcessedMarker) {
      return parseOrThrow(FIRST_RUN_DEFAULT_RANGE);
    } else {
      return lastProcessedMarker;
    }
  };
  const updateLastFullSyncMarker = async (source, lastFullSyncMarker) => {
    await monitoringIndexSourceClient.updateLastFullSyncMarker(source, lastFullSyncMarker);
  };
  const getLastFullSyncMarker = async source => {
    const fromSO = await monitoringIndexSourceClient.getLastFullSyncMarker(source);
    if (fromSO) return fromSO;
    return undefined;
  };
  const findLastEventMarkerInIndex = async (source, action) => {
    var _source$integrations, _resp$hits$hits$0$_so, _resp$hits, _resp$hits$hits, _resp$hits$hits$, _resp$hits$hits$$_sou;
    // find the last event marker in the index
    const esClient = deps.clusterClient.asCurrentUser;
    const resp = await esClient.search({
      index: (_source$integrations = source.integrations) === null || _source$integrations === void 0 ? void 0 : _source$integrations.syncMarkerIndex,
      sort: [{
        '@timestamp': {
          order: 'desc'
        }
      }],
      size: 1,
      query: {
        term: {
          'event.action': action
        }
      },
      _source: ['@timestamp']
    });
    const hit = (_resp$hits$hits$0$_so = resp === null || resp === void 0 ? void 0 : (_resp$hits = resp.hits) === null || _resp$hits === void 0 ? void 0 : (_resp$hits$hits = _resp$hits.hits) === null || _resp$hits$hits === void 0 ? void 0 : (_resp$hits$hits$ = _resp$hits$hits[0]) === null || _resp$hits$hits$ === void 0 ? void 0 : (_resp$hits$hits$$_sou = _resp$hits$hits$._source) === null || _resp$hits$hits$$_sou === void 0 ? void 0 : _resp$hits$hits$$_sou['@timestamp']) !== null && _resp$hits$hits$0$_so !== void 0 ? _resp$hits$hits$0$_so : undefined;
    return hit;
  };
  const detectNewFullSync = async source => {
    const lastFullSync = await getLastFullSyncMarker(source);
    const latestCompletedEvent = await findLastEventMarkerInIndex(source, 'completed');
    // No completed event found, no new full sync.
    if (!latestCompletedEvent) return false;

    // Decide if we should update (first marker OR newer than saved)
    const shouldUpdate = !lastFullSync || latestCompletedEvent > lastFullSync;
    if (!shouldUpdate) return false;
    await updateLastFullSyncMarker(source, latestCompletedEvent);
    return true;
  };
  return {
    updateLastProcessedMarker,
    getLastProcessedMarker,
    updateLastFullSyncMarker,
    getLastFullSyncMarker,
    detectNewFullSync,
    findLastEventMarkerInIndex
  };
};
exports.createSyncMarkersService = createSyncMarkersService;