"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.startEntityStoreDataViewRefreshTask = exports.runEntityStoreDataViewRefreshTask = exports.removeEntityStoreDataViewRefreshTask = exports.registerEntityStoreDataViewRefreshTask = exports.getEntityStoreDataViewRefreshTaskState = void 0;
var _moment = _interopRequireDefault(require("moment"));
var _server = require("@kbn/core/server");
var _entity_store_data_client = require("../../entity_store_data_client");
var _api_key = require("../../auth/api_key");
var _entity_store = require("../../../../../../common/api/entity_analytics/entity_store");
var _state = require("./state");
var _constants = require("./constants");
var _events = require("../../../../telemetry/event_based/events");
var _utils = require("../utils");
var _data_client = require("../../../privilege_monitoring/engine/data_client");
var _indices = require("../../../privilege_monitoring/engine/elasticsearch/indices");
var _check_and_init_privmon_resources = require("../../../privilege_monitoring/check_and_init_privmon_resources");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getTaskName = () => _constants.TYPE;
const getTaskId = namespace => `${_constants.TYPE}:${namespace}:${_constants.VERSION}`;
const registerEntityStoreDataViewRefreshTask = ({
  getStartServices,
  logger,
  telemetry,
  appClientFactory,
  taskManager,
  auditLogger,
  entityStoreConfig,
  experimentalFeatures,
  kibanaVersion,
  isServerless
}) => {
  if (!taskManager) {
    logger.info('[Entity Store] Task Manager is unavailable; skipping entity store data view refresh.');
    return;
  }
  const refreshDataViews = async namespace => {
    const [core, {
      dataViews,
      taskManager: taskManagerStart,
      security,
      encryptedSavedObjects
    }] = await getStartServices();
    const apiKeyManager = (0, _api_key.getApiKeyManager)({
      core,
      logger,
      security,
      encryptedSavedObjects,
      namespace
    });
    const apiKey = await apiKeyManager.getApiKey();
    if (!apiKey) {
      logger.info(`[Entity Store] No API key found, skipping data view refresh in ${namespace} namespace`);
      return;
    }
    const {
      clusterClient,
      soClient
    } = await apiKeyManager.getClientFromApiKey(apiKey);
    const internalUserClient = core.elasticsearch.client.asInternalUser;
    const dataViewsService = await dataViews.dataViewsServiceFactory(soClient, internalUserClient);
    const request = await apiKeyManager.getRequestFromApiKey(apiKey);
    const appClient = appClientFactory.create(request);
    const entityStoreClient = new _entity_store_data_client.EntityStoreDataClient({
      namespace,
      clusterClient,
      soClient,
      logger,
      appClient,
      taskManager: taskManagerStart,
      experimentalFeatures,
      auditLogger,
      telemetry,
      kibanaVersion,
      dataViewsService,
      config: entityStoreConfig,
      security,
      request,
      uiSettingsClient: core.uiSettings.asScopedToClient(soClient),
      isServerless
    });

    // as we have added the privmon index to the list of indices
    // for existing customers, this index may not exist so we need to create it
    const privmonDataClient = new _data_client.PrivilegeMonitoringDataClient({
      logger,
      clusterClient,
      namespace,
      savedObjects: core.savedObjects,
      taskManager: taskManagerStart,
      auditLogger,
      kibanaVersion,
      telemetry,
      experimentalFeatures
    });
    const privmonIndexService = (0, _indices.createPrivmonIndexService)(privmonDataClient);
    await (0, _check_and_init_privmon_resources.checkandInitPrivilegeMonitoringResourcesNoContext)(privmonIndexService, logger);
    const {
      errors
    } = await entityStoreClient.applyDataViewIndices();
    if (errors.length > 0) {
      logger.error(`Errors applying data view changes to the entity store. Errors: \n${errors.join('\n\n')}`);
    }
  };
  taskManager.registerTaskDefinitions({
    [getTaskName()]: {
      title: 'Entity Analytics Entity Store - Execute Data View Refresh Task',
      timeout: _constants.TIMEOUT,
      stateSchemaByVersion: _state.stateSchemaByVersion,
      createTaskRunner: createEntityStoreDataViewRefreshTaskRunnerFactory({
        logger,
        telemetry,
        refreshDataViews,
        experimentalFeatures
      })
    }
  });
};
exports.registerEntityStoreDataViewRefreshTask = registerEntityStoreDataViewRefreshTask;
const startEntityStoreDataViewRefreshTask = async ({
  logger,
  namespace,
  taskManager
}) => {
  const taskId = getTaskId(namespace);
  const log = (0, _utils.entityStoreTaskLogFactory)(logger, taskId);
  log('attempting to schedule');
  try {
    await taskManager.ensureScheduled({
      id: taskId,
      taskType: getTaskName(),
      scope: _constants.SCOPE,
      schedule: {
        interval: _constants.INTERVAL
      },
      state: {
        ..._state.defaultState,
        namespace
      },
      params: {
        version: _constants.VERSION
      }
    });
  } catch (e) {
    logger.warn(`[Entity Store]  [task ${taskId}]: error scheduling task, received ${e.message}`);
    throw e;
  }
};
exports.startEntityStoreDataViewRefreshTask = startEntityStoreDataViewRefreshTask;
const removeEntityStoreDataViewRefreshTask = async ({
  logger,
  namespace,
  taskManager
}) => {
  try {
    await taskManager.remove(getTaskId(namespace));
    logger.info(`[Entity Store] Removed entity store data view refresh task for namespace ${namespace}`);
  } catch (err) {
    if (!_server.SavedObjectsErrorHelpers.isNotFoundError(err)) {
      logger.error(`[Entity Store] Failed to remove entity store data view refresh task: ${err.message}`);
      throw err;
    }
  }
};
exports.removeEntityStoreDataViewRefreshTask = removeEntityStoreDataViewRefreshTask;
const runEntityStoreDataViewRefreshTask = async ({
  refreshDataViews,
  isCancelled,
  logger,
  taskInstance,
  telemetry,
  experimentalFeatures
}) => {
  const state = taskInstance.state;
  const taskId = taskInstance.id;
  const log = (0, _utils.entityStoreTaskLogFactory)(logger, taskId);
  const debugLog = (0, _utils.entityStoreTaskDebugLogFactory)(logger, taskId);
  try {
    const taskStartTime = (0, _moment.default)().utc().toISOString();
    log('running task');
    const updatedState = {
      lastExecutionTimestamp: taskStartTime,
      namespace: state.namespace,
      runs: state.runs + 1
    };
    if (taskId !== getTaskId(state.namespace)) {
      log('outdated task; exiting');
      return {
        state: updatedState
      };
    }
    const start = Date.now();
    debugLog(`Executing data view refresh`);
    try {
      await refreshDataViews(state.namespace);
      log(`Executed data view refresh in ${Date.now() - start}ms`);
    } catch (e) {
      log(`Error executing data view refresh: ${e.message}`);
    }
    const taskCompletionTime = (0, _moment.default)().utc().toISOString();
    const taskDurationInSeconds = (0, _moment.default)(taskCompletionTime).diff((0, _moment.default)(taskStartTime), 'seconds');
    log(`Task run completed in ${taskDurationInSeconds} seconds`);
    telemetry.reportEvent(_events.ENTITY_STORE_DATA_VIEW_REFRESH_EXECUTION_EVENT.eventType, {
      duration: taskDurationInSeconds,
      interval: _constants.INTERVAL
    });
    return {
      state: updatedState
    };
  } catch (e) {
    logger.error(`[Entity Store] [task ${taskId}]: error running task, received ${e.message}`);
    throw e;
  }
};
exports.runEntityStoreDataViewRefreshTask = runEntityStoreDataViewRefreshTask;
const createEntityStoreDataViewRefreshTaskRunnerFactory = ({
  logger,
  telemetry,
  refreshDataViews,
  experimentalFeatures
}) => ({
  taskInstance
}) => {
  let cancelled = false;
  const isCancelled = () => cancelled;
  return {
    run: async () => runEntityStoreDataViewRefreshTask({
      refreshDataViews,
      isCancelled,
      logger,
      taskInstance,
      telemetry,
      experimentalFeatures
    }),
    cancel: async () => {
      cancelled = true;
    }
  };
};
const getEntityStoreDataViewRefreshTaskState = async ({
  namespace,
  taskManager
}) => {
  const taskId = getTaskId(namespace);
  try {
    const taskState = await taskManager.get(taskId);
    return {
      id: taskState.id,
      resource: _entity_store.EngineComponentResourceEnum.task,
      installed: true,
      enabled: taskState.enabled,
      status: taskState.status,
      retryAttempts: taskState.attempts,
      nextRun: taskState.runAt,
      lastRun: taskState.state.lastExecutionTimestamp,
      runs: taskState.state.runs
    };
  } catch (e) {
    if (_server.SavedObjectsErrorHelpers.isNotFoundError(e)) {
      return {
        id: taskId,
        installed: false,
        resource: _entity_store.EngineComponentResourceEnum.task
      };
    }
    throw e;
  }
};
exports.getEntityStoreDataViewRefreshTaskState = getEntityStoreDataViewRefreshTaskState;