"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.installAssetsForRuleMonitoring = void 0;
var _server = require("@kbn/core/server");
var _lodash = require("lodash");
var _pRetry = _interopRequireDefault(require("p-retry"));
var _stream = require("stream");
var _dashboard_rule_monitoring = _interopRequireDefault(require("./dashboard_rule_monitoring.json"));
var _data_view_kibana_event_log = _interopRequireDefault(require("./data_view_kibana_event_log.json"));
var _tag_managed = _interopRequireDefault(require("./tag_managed.json"));
var _tag_security_solution = _interopRequireDefault(require("./tag_security_solution.json"));
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const MAX_RETRIES = 2;

/**
 * Installs managed assets for monitoring rules and health of Detection Engine.
 */
const installAssetsForRuleMonitoring = async (savedObjectsImporter, logger, currentSpaceId) => {
  const operation = async attemptCount => {
    logger.debug(`Installing assets for rule monitoring (attempt ${attemptCount})...`);
    const assets = getAssetsForRuleMonitoring(currentSpaceId);

    // The assets are marked as "managed: true" at the saved object level, which in the future
    // should be reflected in the UI for the user. Ticket to track:
    // https://github.com/elastic/kibana/issues/140364
    const importResult = await savedObjectsImporter.import({
      readStream: _stream.Readable.from(assets),
      managed: true,
      overwrite: true,
      createNewCopies: false,
      refresh: false,
      namespace: spaceIdToNamespace(currentSpaceId)
    });
    importResult.warnings.forEach(w => {
      logger.warn(w.message);
    });
    if (!importResult.success) {
      var _importResult$errors;
      const errors = ((_importResult$errors = importResult.errors) !== null && _importResult$errors !== void 0 ? _importResult$errors : []).map(e => `Couldn't import "${e.type}:${e.id}": ${JSON.stringify(e.error)}`);
      errors.forEach(e => {
        logger.error(e);
      });

      // This will retry the operation
      throw new Error(errors.length > 0 ? errors[0] : `Unknown error (attempt ${attemptCount})`);
    }
    logger.debug('Assets for rule monitoring installed');
  };
  await (0, _pRetry.default)(operation, {
    retries: MAX_RETRIES
  });
};
exports.installAssetsForRuleMonitoring = installAssetsForRuleMonitoring;
const getAssetsForRuleMonitoring = currentSpaceId => {
  const withSpaceId = appendSpaceId(currentSpaceId);
  const assetRuleMonitoringDashboard = (0, _lodash.cloneDeep)(_dashboard_rule_monitoring.default);
  const assetKibanaEventLogDataView = (0, _lodash.cloneDeep)(_data_view_kibana_event_log.default);
  const assetManagedTag = (0, _lodash.cloneDeep)(_tag_managed.default);
  const assetSecuritySolutionTag = (0, _lodash.cloneDeep)(_tag_security_solution.default);

  // Update ids of the assets to include the current space id
  assetRuleMonitoringDashboard.id = withSpaceId('security-detection-rule-monitoring');
  assetManagedTag.id = withSpaceId('fleet-managed');
  assetSecuritySolutionTag.id = withSpaceId('security-solution');

  // Update saved object references of the dashboard accordingly
  assetRuleMonitoringDashboard.references = assetRuleMonitoringDashboard.references.map(reference => {
    if (reference.id === 'fleet-managed-<spaceId>') {
      return {
        ...reference,
        id: assetManagedTag.id
      };
    }
    if (reference.id === 'security-solution-<spaceId>') {
      return {
        ...reference,
        id: assetSecuritySolutionTag.id
      };
    }
    return reference;
  });
  return [assetManagedTag, assetSecuritySolutionTag, assetKibanaEventLogDataView, assetRuleMonitoringDashboard];
};
const appendSpaceId = spaceId => str => `${str}-${spaceId}`;
const spaceIdToNamespace = _server.SavedObjectsUtils.namespaceStringToId;