"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerAssetInventoryUsageCollector = registerAssetInventoryUsageCollector;
var _schema = require("../schema");
var _entities_stats_collector = require("./entities_stats_collector");
var _entities_type_stats_collector = require("./entities_type_stats_collector");
var _asset_criticality_stats_collector = require("./asset_criticality_stats_collector");
var _entity_source_stats_collector = require("./entity_source_stats_collector");
var _entity_store_stats_collector = require("./entity_store_stats_collector");
var _helper = require("../helper");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function registerAssetInventoryUsageCollector(logger, coreServices, usageCollection) {
  // usageCollection is an optional dependency, so make sure to return if it is not registered
  if (!usageCollection) {
    return;
  }

  // Create usage collector
  const assetInventoryUsageCollector = usageCollection.makeUsageCollector({
    type: 'asset_inventory',
    isReady: async () => {
      await coreServices;
      return true;
    },
    fetch: async collectorFetchContext => {
      const awaitPromiseSafe = async (taskName, promise) => {
        try {
          const val = await promise;
          logger.info(`Asset Inventory telemetry: ${taskName} payload was sent successfully`);
          return val;
        } catch (error) {
          logger.error(`${taskName} task failed: ${error.message}`);
          logger.error(error.stack);
          return error;
        }
      };
      const esClient = collectorFetchContext.esClient;
      const [entitiesStats, entitiesTypeStats, entityStoreStats, entitySourceStats, assetCriticalityStats] = await Promise.all([awaitPromiseSafe('Entities', (0, _entities_stats_collector.getEntityStats)(esClient, _helper.ENTITY_INDEX, logger)), awaitPromiseSafe('Entities Type', (0, _entities_type_stats_collector.getEntitiesTypeStats)(esClient, logger)), awaitPromiseSafe('Entity Store', (0, _entity_store_stats_collector.getEntityStoreStats)(esClient, logger)), awaitPromiseSafe('Entity Source', (0, _entity_source_stats_collector.getEntitySourceStats)(esClient, logger)), awaitPromiseSafe('Asset Criticality', (0, _asset_criticality_stats_collector.getAssetCriticalityStats)(esClient, logger))]);
      return {
        entities: entitiesStats,
        entities_type_stats: entitiesTypeStats,
        entity_store_stats: entityStoreStats,
        entity_source_stats: entitySourceStats,
        asset_criticality_stats: assetCriticalityStats
      };
    },
    schema: _schema.assetInventoryUsageSchema
  });

  // Register usage collector
  usageCollection.registerCollector(assetInventoryUsageCollector);
}