"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.KNOWLEDGE_BASE_RETRIEVAL_TOOL = void 0;
var _tools = require("@langchain/core/tools");
var _zod = require("@kbn/zod");
var _document = require("langchain/document");
var _elasticAssistantCommon = require("@kbn/elastic-assistant-common");
var _common = require("../../../../common");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const toolDetails = {
  // note: this description is overwritten when `getTool` is called
  // local definitions exist ../elastic_assistant/server/lib/prompt/tool_prompts.ts
  // local definitions can be overwritten by security-ai-prompt integration definitions
  description: `Call this tool to fetch information from the user's knowledge base. The knowledge base contains useful details the user has saved between conversation contexts.

Use this tool **only in the following cases**:

1. When the user asks a question about their personal, organizational, saved, or previously provided information/knowledge, such as:
- "What was the detection rule I saved for unusual AWS API calls?"
- "Using my saved investigation notes, what did I find about the incident last Thursday?"
- "What are my preferred index patterns?"
- "What did I say about isolating hosts?"
- "What is my favorite coffee spot near the office?" *(non-security example)*

2. Always call this tool when the user's query includes phrases like:**
- "my favorite"
- "what did I say about"
- "my saved"
- "my notes"
- "my preferences"
- "using my"
- "what do I know about"
- "based on my saved knowledge"

3. When you need to retrieve saved information the user has stored in their knowledge base, whether it's security-related or not.

**Do NOT call this tool if**:
- The \`knowledge history\` section already answers the user's question.
- The user's query is about general knowledge not specific to their saved information.

**When calling this tool**:
- Provide only the user's free-text query as the input, rephrased if helpful to clarify the search intent.
- Format the input as a single, clean line of text.

Example:
- User query: "What did I note about isolating endpoints last week?"
- Tool input: "User notes about isolating endpoints."

If no relevant information is found, inform the user you could not locate the requested information.

**Important**:
- Always check the \`knowledge history\` section first for an answer.
- Only call this tool if the user's query is explicitly about their own saved data or preferences.`,
  id: 'knowledge-base-retrieval-tool',
  name: 'KnowledgeBaseRetrievalTool'
};
const KNOWLEDGE_BASE_RETRIEVAL_TOOL = exports.KNOWLEDGE_BASE_RETRIEVAL_TOOL = {
  ...toolDetails,
  sourceRegister: _common.APP_UI_ID,
  isSupported: params => {
    const {
      kbDataClient,
      isEnabledKnowledgeBase
    } = params;
    return isEnabledKnowledgeBase && kbDataClient != null;
  },
  async getTool(params) {
    if (!this.isSupported(params)) return null;
    const {
      kbDataClient,
      logger,
      contentReferencesStore
    } = params;
    if (kbDataClient == null) return null;
    return (0, _tools.tool)(async input => {
      logger.debug(() => `KnowledgeBaseRetrievalToolParams:input\n ${JSON.stringify(input, null, 2)}`);
      const docs = await kbDataClient.getKnowledgeBaseDocumentEntries({
        query: input.query,
        kbResource: 'user',
        required: false
      });
      return JSON.stringify(docs.map(enrichDocument(contentReferencesStore)));
    }, {
      name: toolDetails.name,
      description: params.description || toolDetails.description,
      schema: _zod.z.object({
        query: _zod.z.string().describe(`Summary of items/things to search for in the knowledge base`)
      }),
      tags: ['knowledge-base']
    });
  }
};
function enrichDocument(contentReferencesStore) {
  return document => {
    if (document.id == null) {
      return document;
    }
    const documentId = document.id;
    const reference = contentReferencesStore.add(p => (0, _elasticAssistantCommon.knowledgeBaseReference)(p.id, document.metadata.name, documentId));
    return new _document.Document({
      ...document,
      metadata: {
        ...document.metadata,
        citation: (0, _elasticAssistantCommon.contentReferenceBlock)(reference)
      }
    });
  };
}